#include <boost/python/def.hpp>
#include <boost/python/class.hpp>
#include <boost/python/enum.hpp>
#include <boost/python/init.hpp>
#include <boost/python/manage_new_object.hpp>
#include <boost/python/copy_const_reference.hpp>
#include <boost/python/return_value_policy.hpp>

#include "yandex/blackbox/blackbox2.h"
#include "python_response.h"

namespace py = boost::python;

namespace bb
{

Response* getBulkUser(const BulkResponse* resp, int i) {
    return resp->user(i).release();
}

SessionResp* getSessUser(const MultiSessionResp* resp, int i) {
    return resp->user(i).release();
}

void register_responses()
{
    // Plain response
    py::class_<Response, boost::noncopyable> resp ( "Resp", "Generic blackbox response data", py::no_init );
    resp.def ( "message", &Response::message, py::return_value_policy<py::copy_const_reference>() );
    resp.def ( "__str__", &printInfoResponse );

    // Bulk response
    py::class_<BulkResponse, py::bases<Response>, boost::noncopyable> bresp ( "BulkResp", "Bulk userinfo blackbox response data", py::no_init );
    bresp.def ( "message", &BulkResponse::message, py::return_value_policy<py::copy_const_reference>() );
    bresp.def ( "__str__", &printBulkResponse );
    bresp.def ( "count", &BulkResponse::count );
    bresp.def ( "id", &BulkResponse::id );
    bresp.def ( "user", &getBulkUser, py::return_value_policy<py::manage_new_object>() );

    // login response
    py::enum_<LoginResp::Status> lstat ( "LoginStatus" );
    lstat.value ( "Valid",        LoginResp::Valid );
    lstat.value ( "Disabled",     LoginResp::Disabled );
    lstat.value ( "Invalid",      LoginResp::Invalid );
    lstat.value ( "ShowCaptcha",  LoginResp::ShowCaptcha );
    lstat.value ( "AlienDomain",  LoginResp::AlienDomain );
    lstat.value ( "Compromised",  LoginResp::Compromised );
    lstat.value ( "Expired",      LoginResp::Expired );

    py::enum_<LoginResp::AccountStatus> lastat ( "AccountStatus" );
    lastat.value ( "Unknown",    LoginResp::accUnknown );
    lastat.value ( "Valid",      LoginResp::accValid );
    lastat.value ( "AlienDomain",LoginResp::accAlienDomain );
    lastat.value ( "NotFound",   LoginResp::accNotFound );
    lastat.value ( "Disabled",   LoginResp::accDisabled );

    py::enum_<LoginResp::PasswdStatus> lpstat ( "PasswdStatus" );
    lpstat.value ( "Unknown",    LoginResp::pwdUnknown );
    lpstat.value ( "Valid",      LoginResp::pwdValid );
    lpstat.value ( "Bad",        LoginResp::pwdBad );
    lpstat.value ( "Compromised",LoginResp::pwdCompromised );

    py::class_<LoginResp, py::bases<Response>, boost::noncopyable> lresp ( "LoginResp", "Blackbox login response data", py::no_init );
    lresp.def ( "status", &LoginResp::status );
    lresp.def ( "accStatus", &LoginResp::accStatus );
    lresp.def ( "pwdStatus", &LoginResp::pwdStatus );
    lresp.def ( "__str__", &printLoginResponse );

    // session response
    py::enum_<SessionResp::Status> sstat ( "SessionStatus" );
    sstat.value ( "Valid",        SessionResp::Valid );
    sstat.value ( "NeedReset",    SessionResp::NeedReset );
    sstat.value ( "Expired",      SessionResp::Expired );
    sstat.value ( "NoAuth",       SessionResp::NoAuth );
    sstat.value ( "Disabled",     SessionResp::Disabled );
    sstat.value ( "Invalid",      SessionResp::Invalid );
    sstat.value ( "WrongGuard",   SessionResp::WrongGuard );

    py::class_<SessionResp, py::bases<Response>, boost::noncopyable> sresp ( "SessionResp", "Blackbox session response data", py::no_init );
    sresp.def ( "status", &SessionResp::status );
    sresp.def ( "age", &SessionResp::age, py::return_value_policy<py::copy_const_reference>() );
    sresp.def ( "isLite", &SessionResp::isLite );
    sresp.def ( "__str__", &printSessionResponse );

    // multisession response
    py::class_<MultiSessionResp, py::bases<Response>, boost::noncopyable> msresp ( "MultiSessionResp", "Blackbox session response data", py::no_init );
    msresp.def ( "status", &MultiSessionResp::status );
    msresp.def ( "age", &MultiSessionResp::age, py::return_value_policy<py::copy_const_reference>() );
    msresp.def ( "defaultUid", &MultiSessionResp::defaultUid, py::return_value_policy<py::copy_const_reference>() );
    msresp.def ( "__str__", &printMultiSessionResponse );
    msresp.def ( "count", &MultiSessionResp::count );
    msresp.def ( "id", &MultiSessionResp::id );
    msresp.def ( "user", &getSessUser, py::return_value_policy<py::manage_new_object>() );

    // mailhost response
    py::enum_<HostResp::Status> hstat ( "HostStatus" );
    hstat.value ( "OK",               HostResp::OK );
    hstat.value ( "AccessDenied",     HostResp::AccessDenied );
    hstat.value ( "MissingParam",     HostResp::MissingParam );
    hstat.value ( "UnknownOp",        HostResp::UnknownOp );
    hstat.value ( "UnknownScope",     HostResp::UnknownScope );
    hstat.value ( "InvalidParam",     HostResp::InvalidParam );
    hstat.value ( "RecordNotFound",   HostResp::RecordNotFound );

    py::class_<HostResp, boost::noncopyable> hresp ( "HostResp", "Blackbox mailhost response data", py::no_init );
    hresp.def ( "status", &HostResp::status );
    hresp.def ( "message", &HostResp::message, py::return_value_policy<py::copy_const_reference>() );
    hresp.def ( "__str__", &printHostResponse );

    // PwdQuality response
    py::class_<PwdQualityResp, boost::noncopyable> pwdresp ( "PwdQualityResp", "Password quality blackbox response data", py::no_init );
    pwdresp.def ( "message", &PwdQualityResp::message, py::return_value_policy<py::copy_const_reference>() );
    pwdresp.def ( "__str__", &printPwdQualityResponse );

    // response parsers
    py::def ( "Response", &Python_InfoResponse, py::return_value_policy<py::manage_new_object>() );
    py::def ( "BulkResponse", &Python_BulkResponse, py::return_value_policy<py::manage_new_object>() );
    py::def ( "LoginResponse", &Python_LoginResponse, py::return_value_policy<py::manage_new_object>() );
    py::def ( "SessionIDResponse", &Python_SessionIDResponse, py::return_value_policy<py::manage_new_object>() );
    py::def ( "SessionIDResponseMulti", &Python_SessionIDResponseMulti, py::return_value_policy<py::manage_new_object>() );
    py::def ( "MailHostResponse", &Python_HostResponse, py::return_value_policy<py::manage_new_object>() );
    py::def ( "PwdQualityResponse", &Python_PwdQualityResponse, py::return_value_policy<py::manage_new_object>() );
}

} // namespace bb

// vi: expandtab:sw=4:ts=4
// kate: replace-tabs on; indent-width 4; tab-width 4;
