#!/usr/bin/python

# Utility script for testing Python binding
from testcommon import *

def print_common_data ( resp ) :
    out = ""
      # Print common response data
    out += "Response message: '%s'\n" % resp.message()

    regname   = bb.Regname(resp)
    uid       = bb.Uid(resp)
    liteuid   = bb.LiteUid(resp)
    pddinfo   = bb.PDDUserInfo(resp)
    karmainfo = bb.KarmaInfo(resp)
    displayname = bb.DisplayNameInfo(resp)
    logininfo = bb.LoginInfo(resp)

    out += "User info: "

    if regname.value:
        out += "regname='%s', " % regname.value

    if uid.uid:
        out += "uid=%s" % uid.uid

    if liteuid.liteUid:
        out += "liteUid=%s" % liteuid.liteUid
    out += "\n"

    if uid.hosted:
        out += "PDD user info: domainId=%s, domain='%s', mx='%s', domEna='%s', catchAll='%s'\n" % \
                (pddinfo.domId, pddinfo.domain, pddinfo.mx, pddinfo.domEna, pddinfo.catchAll)

    out += "Karma info: karma=%s, banTime='%s', karmaStatus='%s'\n" % \
                (karmainfo.karma, karmainfo.banTime, karmainfo.karmaStatus)
    out += "Login name: '%s', havePassword=%s, haveHint=%s\n" % (logininfo.login,
             "true" if logininfo.have_password else "false",
             "true" if logininfo.have_hint else "false")

    if displayname.name:
        out += "DisplayName: '%s'" % displayname.name
        if displayname.social:
            out += ", social account, provider='%s', profileId=%s, redirectTarget=%s\n" % \
                (displayname.socProvider, displayname.socProfile, displayname.socTarget)
        else: out += '\n'
        if displayname.defaultAvatar:
            out += 'Default avatar: ' + displayname.defaultAvatar + '\n';

    emails = bb.EmailList(resp)
    aliases = bb.AliasList(resp)
    fields = bb.DBFields(resp)
    attributes = bb.Attributes(resp)

    if not emails.empty():
        out += "Emails list: (size=%d) [\n" % emails.size()
        for item in emails.getEmailItems():
            out += "\taddr='%s', born date='%s'" % (item.address, item.date)
            if item.default:
                out += ", default"
            if item.native:
                out += ", native"
            elif item.rpop:
                out += ", remote pop"
            else:
                if not item.validated:
                    out += ", not"
                else: out += ","
                out += " validated"
            out += "\n"
        out += "]\n"

        if emails.getDefault():
            out += "Default email: '%s'\n" % emails.getDefault()

    if not aliases.empty():
        out += "Alias list: (size=%d) [\n" % aliases.size()
        for alias in aliases.getAliases():
            out += "\ttype='%s', alias='%s'\n" % (alias.type, alias.alias)
        out += "]\n"

    if not fields.empty():
        out += "DBFields: (size=%d) [\n" % fields.size()
        for field in fields:
            out += "\t'%s'='%s'\n" % (field.key, field.val)
        out += "]\n"

    if not attributes.empty():
        out += "Attributes: (size=%d) [\n" % attributes.size()
        for attr in attributes:
            out += "\t'%s'='%s'\n" % (attr.key, attr.val)
        out += "]\n"

    return out


def print_response ( resp ) :
    if not resp:
        return "NULL Response."

    # Build result string
    out = ""

    # Print mailhosts response data
    if isinstance(resp, bb.HostResp) :
        out += "Host Response with status: %s and message: '%s'\n" % \
               (resp.status(), resp.message())
        hosts = bb.MailHostsList(resp)
        if not hosts.empty() :
          out += "MailHost list: (size=%d) [\n" % hosts.size()
          for host in hosts.getHosts():
             out += "\thostId='%s', hostIp='%s', hostName='%s', " \
                 "dbId='%s', sid='%s', priority='%s', mx='%s'\n" % \
                 (host.hostId, host.hostIp, host.hostName,
                  host.dbId, host.sid, host.priority, host.mx)
          out += "]\n"
        return out

    # Print pwdquality response data
    if isinstance(resp, bb.PwdQualityResp) :
        out += "Password Quality Response with message: '%s'\n" % resp.message()

        pwd_quality = bb.PasswordQuality(resp)
        if pwd_quality.quality:
            out += "Password quality: %s\n" % pwd_quality.quality
        return out

    # Print bulk response data
    if isinstance(resp, bb.BulkResp) :
        out += "Bulk Response with %d children and message '%s'\n\n" % \
              ( resp.count(), resp.message())
        for i in range(0, resp.count() ) :
          out += "Bulk child #%d with id='%s':\n" % (i, resp.id(i))
          out += print_common_data( resp.user(i) )
          out += "\n"

        out += print_common_data( resp )
        return out

    # Print login response data
    if isinstance(resp, bb.LoginResp) :
        out += "Login Response with status: %s\n" % resp.status()
        out += "Account status: %s, password status: %s\n" % \
            ( resp.accStatus(), resp.pwdStatus() )
        policy = bb.BruteforcePolicy(resp)

        if policy.mode!=bb.ResistMode.None :
          out += "Bruteforce policy: mode=%s" % policy.mode
          if policy.mode==bb.ResistMode.Delay :
            out += ", delay=%s" % policy.delay
          out += '\n'

        pwd_quality = bb.PasswordQuality(resp)
        if pwd_quality.quality:
            out += "Password quality: %s\n" % pwd_quality.quality

        connection_id = bb.ConnectionId(resp)
        if connection_id.connectionId:
            out += "Connection Id: %s\n" % connection_id.connectionId

    # Print session response data
    if isinstance ( resp, bb.SessionResp ) :
        out += "Session Response with status: %s\n" % resp.status()
        out += "Session type: %s, age: %s\n" % ("Lite" if resp.isLite() else "Full", resp.age())

        newid = bb.NewSessionId(resp)
        if newid.value:
            out += "New sessionId='%s', domain='%s', expires='%s', httpOnly=%s\n" % \
              ( newid.value, newid.domain, newid.expires, "true" if newid.httpOnly else "false" )

    # Print multisession response data
    if isinstance ( resp, bb.MultiSessionResp ) :
        out += "MultiSession Response with %d children and status: %s\n" % (resp.count(), resp.status())
        out += "Session age: %s\n" % resp.age()
        out += "Response message: '%s'\n" % resp.message()
        out += "Default uid: %s\n" % resp.defaultUid()

        newid = bb.NewSessionId(resp)
        if newid.value:
            out += "New sessionId='%s', domain='%s', expires='%s', httpOnly=%s\n" % \
              ( newid.value, newid.domain, newid.expires, "true" if newid.httpOnly else "false" )

        for i in range(0, resp.count() ) :
          out += "Multisession child #%d with id='%s':\n" % (i, resp.id(i))
          out += print_response( resp.user(i) )
          out += "\n"

        kind = bb.SessionKind(resp)
        authid = bb.AuthId(resp)
        connection_id = bb.ConnectionId(resp)

        if kind.kind != bb.SessKind.None :
          out += "Session kind: %s ( '%s' )\n" % (kind.kind, kind.kindName)

        if authid.authId:
          out += "AuthId: '%s', time=%s, ip=%s, host=%s\n" % (authid.authId, authid.time, authid.userIp, authid.hostId)

        more_users = bb.AllowMoreUsers(resp)

        out += "Allow more users: %s\n" % "true" if more_users.allowMoreUsers else "false"

        if connection_id.connectionId:
            out += "Connection Id: %s\n" % connection_id.connectionId

        return out

    out += print_common_data(resp)

    # Print session response additional data
    if isinstance ( resp, bb.SessionResp ) :
        ainfo = bb.AuthInfo(resp)
        oinfo = bb.OAuthInfo(resp)
        kind = bb.SessionKind(resp)
        authid = bb.AuthId(resp)
        connection_id = bb.ConnectionId(resp)

        if ainfo.age:
            out += "Password verified %s sec ago" % ainfo.age
            if ainfo.social:
                out += ", social authorization, profile id=%s" % ainfo.profileId
            out += ", session %ssecure" % ( "in" if not ainfo.secure else "")
            out += "\n"

        if not oinfo.empty():
          out += "OAuth info: (size=%d) [" % oinfo.size()
          first = True
          for info in oinfo.getInfo():
            if first:
                first = False
            else: out += ","
            out += " '%s':'%s'" % (info.key, info.val)
          out += " ]\n"

        if kind.kind != bb.SessKind.None :
          out += "Session kind: %s ( '%s' )\n" % (kind.kind, kind.kindName)

        if authid.authId:
          out += "AuthId: '%s', time=%s, ip=%s, host=%s\n" % (authid.authId, authid.time, authid.userIp, authid.hostId)

        if connection_id.connectionId:
            out += "Connection Id: %s\n" % connection_id.connectionId

    return out


if __name__ == '__main__':
    if len(sys.argv) < 2 :
        print "Usage:", sys.argv[0], " resp|bulk|login|session|multisession|host|pwdquality [filename]"
        sys.exit(1)

    resp_type = sys.argv[1]

    if resp_type not in ["resp", "bulk", "login", "session", "multisession", "host", "pwdquality" ] :
        print "Unknown response type:", resp_type
        sys.exit(2)

    if len(sys.argv) > 2 and sys.argv[2] :
        f = open( sys.argv[2] , 'r')
    else:
        f = sys.stdin

    resp_text = f.read()

    try:
        if resp_type == "login" :
            response = bb.LoginResponse(resp_text)
        elif resp_type == "session" :
            response = bb.SessionIDResponse(resp_text)
        elif resp_type == "multisession" :
            response = bb.SessionIDResponseMulti(resp_text)
        elif resp_type == "host" :
            response = bb.MailHostResponse(resp_text)
        elif resp_type == "bulk" :
            response = bb.BulkResponse(resp_text)
        elif resp_type == "pwdquality" :
            response = bb.PwdQualityResponse(resp_text)
        else:
            response = bb.Response(resp_text)

        # Print parsed response using libBlackbox code, to double check
        cpp_response_printed = str(response)

        response_printed = print_response(response)

        if cpp_response_printed != response_printed :
            print "Python and C++ printed responses differ!"

        #don't use print, it prints newline even if , added
        sys.stdout.write( response_printed )
    except bb.FatalError, e:
        print "Got a fatal blackbox exception: code=%s, message='%s'" % (e.code(), e.what())
    except bb.TempError, e:
        print "Got a temporary blackbox exception: code=%s, message='%s'" % (e.code(), e.what())
    except:
        print "Error: Got unknown exception:", str(sys.exc_info()[0]), str(sys.exc_info()[1]), str(sys.exc_info()[2])
        sys.exit(4)
