// ====================================================================
//  Implementation of response information accessors
// ====================================================================

#include "yandex/blackbox/accessors.h"

#include "responseimpl.h"
#include "accessors_ext.h"
#include "utils.h"

namespace bb {
    using namespace std;

    inline Response::Impl* getImplChecked( const Response* pResp )
    {
        if ( !pResp )
            throw FatalError(NSessionCodes::INVALID_PARAMS, "NULL Response pointer");

        return pResp->getImpl();
    }

    Regname::Regname(const Response* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("regname", value_);
    }

    const std::string& Regname::value() const
    {
        return value_;
    }

    Uid::Uid(const Response* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("uid", uid_);

        long int hosted;
        if ( p->getIfExists("uid/@hosted", hosted) && hosted )
            hosted_ = true;
        else hosted_ = false;
    }

    Uid::~Uid()
    {}

    const std::string& Uid::uid() const
    {
        return uid_;
    }

    bool Uid::hosted() const
    {
        return hosted_;
    }

    LiteUid::LiteUid(const Response* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("liteuid", liteUid_);
    }

    LiteUid::~LiteUid()
    {}

    const std::string& LiteUid::liteUid() const
    {
        return liteUid_;
    }

    bool LiteUid::liteAccount() const
    {
        return liteAccount_;
    }

    PDDUserInfo::PDDUserInfo(const Response* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("uid/@domid", domId_);

        string tmp_domain, decoded;
        p->getIfExists("uid/@domain", tmp_domain);

        if ( !tmp_domain.empty() )
            domain_ = idn::decode(tmp_domain, decoded);

        p->getIfExists("uid/@mx", mx_);

        std::string ena, catchall;
        p->getIfExists("uid/@domain_ena", ena);
        p->getIfExists("uid/@catch_all", catchall);

        if (!ena.empty() || !catchall.empty()) {
            pExt_.reset(new Ext());
            pExt_->domEna_.swap(ena);
            pExt_->catchAll_.swap(catchall);
        }
    }

    PDDUserInfo::~PDDUserInfo()
    {}

    const std::string& PDDUserInfo::domId() const
    {
        return domId_;
    }

    const std::string& PDDUserInfo::domain() const
    {
        return domain_;
    }

    const std::string& PDDUserInfo::mx() const
    {
        return mx_;
    }

    const std::string& PDDUserInfo::domEna() const
    {
        if (pExt_.get())
            return pExt_->domEna_;
        else return emptyStr;
    }

    const std::string& PDDUserInfo::catchAll() const
    {
        if (pExt_.get())
            return pExt_->catchAll_;
        else return emptyStr;
    }


    KarmaInfo::KarmaInfo(const Response* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("karma", karma_);
        p->getIfExists("karma/@allow-until", banTime_);

        std::string k_status;
        if (p->getIfExists("karma_status", k_status) &&
            !k_status.empty()) {
            pExt_.reset(new Ext());
            pExt_->karmaStatus_.swap(k_status);
        }
    }

    KarmaInfo::~KarmaInfo()
    {}

    const std::string& KarmaInfo::karma() const
    {
        return karma_;
    }

    const std::string& KarmaInfo::banTime() const
    {
        return banTime_;
    }

    const std::string& KarmaInfo::karmaStatus() const
    {
        if (pExt_.get())
            return pExt_->karmaStatus_;
        else return emptyStr;
    }

    DisplayNameInfo::DisplayNameInfo(const Response* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("display_name/name", name_);

        string dummy;
        social_ = p->getIfExists("display_name/social", dummy);

        if ( social_ ) {
            p->getIfExists("display_name/social/profile_id", socProfile_);
            p->getIfExists("display_name/social/provider", socProvider_);
            p->getIfExists("display_name/social/redirect_target", socTarget_);
        }

        std::string def_avatar;
        if (p->getIfExists("display_name/avatar/default", def_avatar) &&
            !def_avatar.empty()) {
            pExt_.reset(new Ext());
            pExt_->defaultAvatar_.swap(def_avatar);
        }
    }

    DisplayNameInfo::~DisplayNameInfo()
    {}

    const std::string& DisplayNameInfo::name() const
    {
        return name_;
    }

    bool DisplayNameInfo::social() const
    {
        return social_;
    }

    const std::string& DisplayNameInfo::socProfile() const
    {
        return socProfile_;
    }

    const std::string& DisplayNameInfo::socProvider() const
    {
        return socProvider_;
    }

    const std::string& DisplayNameInfo::socTarget() const
    {
        return socTarget_;
    }

    const std::string& DisplayNameInfo::defaultAvatar() const
    {
        if (pExt_.get())
            return pExt_->defaultAvatar_;
        else return emptyStr;
    }

    LoginInfo::LoginInfo(const Response* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("login", login_);

        long int passwd;
        if ( p->getIfExists("have_password", passwd) && passwd )
            havePassword_ = true;
        else havePassword_ = false;

        bool have_hint;
        long int hint;
        if ( p->getIfExists("have_hint", hint) && hint )
            have_hint = true;
        else have_hint = false;

        pExt_.reset(new Ext());
        pExt_->haveHint_ = have_hint;
    }

    LoginInfo::~LoginInfo()
    {}

    const std::string& LoginInfo::login() const
    {
        return login_;
    }

    bool LoginInfo::havePassword() const
    {
        return havePassword_;
    }

    bool LoginInfo::haveHint() const
    {
        if (pExt_.get())
            return pExt_->haveHint_;
        else return false;
    }

    EmailList::Item::Item()
    {}

    EmailList::Item::Item(const Item& i)
        : native_(i.native_), validated_(i.validated_), default_(i.default_),
        rpop_(i.rpop_), address_(i.address_), date_(i.date_), pExt_()
    {
        if ( i.pExt_.get() )
            pExt_.reset( new Ext(*i.pExt_) );
    }

    EmailList::Item::~Item()
    {}

    EmailList::Item& EmailList::Item::operator=(const EmailList::Item& i)
    {
        native_ = i.native_;
        validated_ = i.validated_;
        default_ = i.default_;
        rpop_ = i.rpop_;
        address_ = i.address_;
        date_ = i.date_;

        if ( i.pExt_.get() )
            pExt_.reset(new Ext(*i.pExt_));
        else pExt_.reset();

        return *this;
    }

    bool EmailList::Item::operator==(const EmailList::Item& i)
    {
        return address_ == i.address_;
    }

    bool EmailList::Item::native() const
    {
        return native_;
    }

    bool EmailList::Item::validated() const
    {
        return validated_;
    }

    bool EmailList::Item::isDefault() const
    {
        return default_;
    }

    bool EmailList::Item::rpop() const
    {
        return rpop_;
    }

    const std::string& EmailList::Item::address() const
    {
        return address_;
    }

    const std::string& EmailList::Item::date() const
    {
        return date_;
    }

    EmailList::EmailList(const Response* pResp)
        : pDefault_(NULL)
    {
        Response::Impl* p = getImplChecked( pResp );

        xmlConfig::Parts mails ( p->getParts("address-list/address") );
        list_.resize( mails.Size() );

        long int val;
        for (int i=0; i < mails.Size(); ++i)
        {
            xmlConfig::Part item ( mails[i] );

            string tmp_addr(item.asString()), decoded;

            list_[i].address_ = idn::decodeAddr(tmp_addr, decoded);

            item.GetIfExists("@born-date", list_[i].date_ );

            list_[i].native_ = item.GetIfExists("@native", val) && val;
            list_[i].validated_ = item.GetIfExists("@validated", val) && val;
            list_[i].rpop_ = item.GetIfExists("@rpop", val) && val;

            if ( item.GetIfExists("@default", val) && val ) {
                list_[i].default_ = true;
                pDefault_ = &list_[i];
            }
            else list_[i].default_ = false;
        }
    }

namespace {
    void readCookieData(const Response* pResp, const std::string& name,
        std::string& value, std::string& domain, std::string& expires, bool& httpOnly) {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists(name, value);
        p->getIfExists(name+"/@domain", domain);
        p->getIfExists(name+"/@expires", expires);

        long int val;
        httpOnly = p->getIfExists(name+"/@http-only", val) && val;
    }
}
    NewSessionId::NewSessionId(const SessionResp* pResp)
    {
        readCookieData(pResp, "new-session", value_, domain_, expires_, http_only_);
    }

    NewSessionId::NewSessionId(const MultiSessionResp* pResp)
    {
        readCookieData(pResp, "new-session", value_, domain_, expires_, http_only_);
    }

    const std::string& NewSessionId::value() const
    {
        return value_;
    }

    const std::string& NewSessionId::domain() const
    {
        return domain_;
    }

    const std::string& NewSessionId::expires() const
    {
        return expires_;
    }

    bool NewSessionId::httpOnly() const
    {
        return http_only_;
    }

    AuthInfo::AuthInfo(const SessionResp* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("auth/password_verification_age", age_);

        string dummy;
        social_ = p->getIfExists("auth/social", dummy);

        if ( social_ )
            p->getIfExists("auth/social/profile_id", profileId_);

        long int val;
        secure_ = p->getIfExists("auth/secure", val) && val;
    }

    AuthInfo::~AuthInfo()
    {}

    bool AuthInfo::social() const
    {
        return social_;
    }

    const std::string& AuthInfo::age() const
    {
        return age_;
    }

    const std::string& AuthInfo::profileId() const
    {
        return profileId_;
    }

    bool AuthInfo::secure() const
    {
        return secure_;
    }

    AllowMoreUsers::AllowMoreUsers(const MultiSessionResp* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        long int val;
        allowMoreUsers_ = p->getIfExists("allow_more_users", val) && val;
    }

    AllowMoreUsers::~AllowMoreUsers()
    {}

    bool AllowMoreUsers::allowMoreUsers() const
    {
        return allowMoreUsers_;
    }

    AliasList::Item::Item()
    {}

    AliasList::Item::Item(const Item& i)
        : type_(i.type_), alias_(i.alias_)
    {
        if ( i.pExt_.get() )
            pExt_.reset( new Ext(*i.pExt_) );
    }

    AliasList::Item::~Item()
    {}

    AliasList::Item& AliasList::Item::operator=(const Item& i)
    {
        type_ = i.type_;
        alias_ = i.alias_;

        if ( i.pExt_.get() )
            pExt_.reset(new Ext(*i.pExt_));
        else pExt_.reset();

        return *this;
    }

    bool AliasList::Item::operator==(const Item& i)
    {
        return (type_ == i.type_) && (alias_ == i.alias_);
    }

    AliasList::Item::Type AliasList::Item::type() const
    {
        return type_;
    }

    const std::string& AliasList::Item::alias() const
    {
        return alias_;
    }

    AliasList::AliasList(const Response* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        xmlConfig::Parts aliases ( p->getParts("aliases/alias") );
        list_.resize( aliases.Size() );

        long int val;
        for (int i=0; i < aliases.Size(); ++i)
        {
            xmlConfig::Part item ( aliases[i] );

            val = 0;
            item.GetIfExists("@type", val);

            list_[i].type_ = static_cast<Item::Type>(val);
            list_[i].alias_ = item.asString();
        }
    }

    MailHostsList::Item::Item()
    {}

    MailHostsList::Item::Item(const Item& i)
        : hostId_(i.hostId_), hostIp_(i.hostIp_), hostName_(i.hostName_),
          dbId_(i.dbId_), sid_(i.sid_), priority_(i.priority_), mx_(i.mx_)
    {
        if ( i.pExt_.get() )
            pExt_.reset( new Ext(*i.pExt_) );
    }

    MailHostsList::Item::~Item()
    {}

    MailHostsList::Item& MailHostsList::Item::operator=(const Item& i)
    {
        hostId_ = i.hostId_;
        hostIp_ = i.hostIp_;
        hostName_ = i.hostName_;
        dbId_ = i.dbId_;
        sid_ = i.sid_;
        priority_ = i.priority_;
        mx_ = i.mx_;

        if ( i.pExt_.get() )
            pExt_.reset(new Ext(*i.pExt_));
        else pExt_.reset();

        return *this;
    }

    bool MailHostsList::Item::operator==(const Item& i)
    {
        return (hostId_ == i.hostId_);
    }

    const std::string& MailHostsList::Item::hostId() const
    {
        return hostId_;
    }

    const std::string& MailHostsList::Item::hostIp() const
    {
        return hostIp_;
    }

    const std::string& MailHostsList::Item::hostName() const
    {
        return hostName_;
    }

    const std::string& MailHostsList::Item::dbId() const
    {
        return dbId_;
    }

    const std::string& MailHostsList::Item::sid() const
    {
        return sid_;
    }

    const std::string& MailHostsList::Item::priority() const
    {
        return priority_;
    }

    const std::string& MailHostsList::Item::mx() const
    {
        return mx_;
    }

    MailHostsList::MailHostsList(const HostResp* pResp)
    {
        if ( !pResp )
            throw FatalError(NSessionCodes::INVALID_PARAMS, "NULL Response pointer");

        Response::Impl* p = pResp->getImpl();

        xmlConfig::Parts hosts ( p->getParts("entry") );
        list_.resize( hosts.Size() );

        for (int i=0; i < hosts.Size(); ++i)
        {
            xmlConfig::Part item ( hosts[i] );

            item.GetIfExists("@hostd_id",   list_[i].hostId_);
            item.GetIfExists("@host_ip",    list_[i].hostIp_);
            item.GetIfExists("@host_name",  list_[i].hostName_);
            item.GetIfExists("@db_id",      list_[i].dbId_);
            item.GetIfExists("@sid",        list_[i].sid_);
            item.GetIfExists("@prio",       list_[i].priority_);
            item.GetIfExists("@mx",         list_[i].mx_);
        }
    }

    OAuthInfo::OAuthInfo(const SessionResp* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        xmlConfig::Parts oauth ( p->getParts("OAuth/*") );

        for (int i=0; i < oauth.Size(); ++i)
        {
            xmlConfig::Part item ( oauth[i] );
            info_[item.GetName()] = item.asString();
        }
    }

    BruteforcePolicy::BruteforcePolicy(const LoginResp* pResp) :
        mode_(None), delay_(0)
    {
        Response::Impl* p = getImplChecked( pResp );

        long int val;

        string tmp;
        if ( p->getIfExists("bruteforce_policy/captcha", tmp ) )
            mode_ = Captcha;
        else if ( p->getIfExists("bruteforce_policy/password_expired", tmp ) )
            mode_ = Expired;
        else if ( p->getIfExists("bruteforce_policy/delay", val ) ) {
            mode_ = Delay;
            delay_ = static_cast<int>(val);
        }
    }

    BruteforcePolicy::~BruteforcePolicy()
    {}

    int BruteforcePolicy::delay() const
    {
        return delay_;
    }

    BruteforcePolicy::Mode BruteforcePolicy::mode() const
    {
        return mode_;
    }

    SessionKind::SessionKind(const SessionResp* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        long int k;

        if ( p->getIfExists("special/@id", k) ) {
            kind_ = static_cast<Kind>(k);
            p->getIfExists("special", kindName_);
        }
        else kind_ = None;
    }

    SessionKind::SessionKind(const MultiSessionResp* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        long int k;

        if ( p->getIfExists("special/@id", k) ) {
            kind_ = static_cast<Kind>(k);
            p->getIfExists("special", kindName_);
        }
        else kind_ = None;
    }

    SessionKind::Kind SessionKind::kind() const
    {
        return kind_;
    }

    const std::string& SessionKind::kindName() const
    {
        return kindName_;
    }

    AuthId::AuthId(const SessionResp* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("authid/@time", time_);
        p->getIfExists("authid/@ip", userIp_);
        p->getIfExists("authid/@host", hostId_);
        p->getIfExists("authid", authId_);
    }

    AuthId::AuthId(const MultiSessionResp* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("authid/@time", time_);
        p->getIfExists("authid/@ip", userIp_);
        p->getIfExists("authid/@host", hostId_);
        p->getIfExists("authid", authId_);
    }

    AuthId::~AuthId()
    {}

    const std::string& AuthId::time() const
    {
        return time_;
    }

    const std::string& AuthId::userIp() const
    {
        return userIp_;
    }

    const std::string& AuthId::hostId() const
    {
        return hostId_;
    }

    const std::string& AuthId::authId() const
    {
        return authId_;
    }

    PasswordQuality::PasswordQuality(const LoginResp* pResp)
    {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("password_quality", quality_);
    }

    PasswordQuality::PasswordQuality(const PwdQualityResp* pResp) {
        if ( !pResp )
            throw FatalError(NSessionCodes::INVALID_PARAMS, "NULL Response pointer");

        Response::Impl* p = pResp->getImpl();

        p->getIfExists("password_quality", quality_);
    }

    PasswordQuality::~PasswordQuality()
    {}

    const std::string& PasswordQuality::quality() const {
        return quality_;
    }

    ConnectionId::ConnectionId(const LoginResp* pResp) {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("connection_id", connectionId_);
    }

    ConnectionId::ConnectionId(const SessionResp* pResp) {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("connection_id", connectionId_);
    }

    ConnectionId::ConnectionId(const MultiSessionResp* pResp) {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("connection_id", connectionId_);
    }

    ConnectionId::~ConnectionId()
    {}

    const std::string& ConnectionId::connectionId() const {
        return connectionId_;
    }

    LoginId::LoginId(const LoginResp* pResp) {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("login_id", loginId_);
    }

    LoginId::LoginId(const SessionResp* pResp) {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("login_id", loginId_);
    }

    LoginId::LoginId(const MultiSessionResp* pResp) {
        Response::Impl* p = getImplChecked( pResp );

        p->getIfExists("login_id", loginId_);
    }

    LoginId::~LoginId()
    {}

    const std::string& LoginId::loginId() const {
        return loginId_;
    }

    UserTicket::UserTicket(const Response* resp) {
        Response::Impl* p = getImplChecked(resp);
        p->getIfExists("user_ticket", userTicket_);
        if (userTicket_.empty()) {
            throw FatalError(NSessionCodes::INVALID_PARAMS, "UserTicket does not exist");
        }
    }

    UserTicket::~UserTicket() {
    }

}   // namespace bb

// vi: expandtab:sw=4:ts=4
// kate: replace-tabs on; indent-width 4; tab-width 4;
