// ====================================================================
//  Implementation of main blackbox interface functions
// ====================================================================

#include <yandex/blackbox/blackbox2.h>

#include "utils.h"
#include "server.h"
#include "responseimpl.h"

#include <cstdio>
#include <ctime>

namespace bb {
    using namespace std;

    // ==== Response structure ============================================

    Response::Response(auto_ptr<Impl> pImpl)
        : pImpl_(pImpl), message_()
    {
        if ( !pImpl_.get() )
            throw FatalError( NSessionCodes::UNKNOWN,
                              "Internal error: NULL Response::Impl" );

        // empty if no <error> tag
        pImpl_->getIfExists("error", message_);

        long int err_code;
        if ( pImpl_->getIfExists("exception/@id", err_code) )
        {
            NSessionCodes::ESessionError code = static_cast<NSessionCodes::ESessionError>(err_code);
            if ( code == NSessionCodes::DB_EXCEPTION ||
                 code == NSessionCodes::DB_FETCHFAILED)
                 throw TempError( code, message_);
            else throw FatalError( code, message_ );
        }
    }

    Response::~Response() {
    }

    BulkResponse::BulkResponse(auto_ptr<Impl> pImpl)
        : Response(pImpl)
    {
        xmlConfig::Parts users = pImpl_->getParts("user");
        count_ = users.Size();
    }

    BulkResponse::~BulkResponse() {
    }

    string BulkResponse::id(int i) const {
        if ( i<0 || i>=count_ )
            throw FatalError( NSessionCodes::INVALID_PARAMS,
                              "Bulk response index out of range" );

        xmlConfig::Part user = pImpl_->getParts("user")[i];

        string id;
        user.GetIfExists("@id", id);

        return id;
    }

    auto_ptr<Response> BulkResponse::user(int i) const {
        if ( i<0 || i>=count_ )
            throw FatalError( NSessionCodes::INVALID_PARAMS,
                              "Bulk response index out of range" );

        xmlConfig::Part user = pImpl_->getParts("user")[i];

        Response* pResp = new Response( auto_ptr<Response::Impl>(
            new Response::Impl(&user) ) );

        return auto_ptr<Response> (pResp);
    }

    ConsumerContext::ConsumerContext(const string& consumer, const string& clientId, const string& base64Secret)
        : consumer_(consumer)
        , clientId_(clientId)
        , secret_  (base64url2bin(base64Secret))
    {
    }

    SessionResp::SessionResp(auto_ptr<Impl> pImpl)
        : Response(pImpl)
    {
        long int status;
        if ( !pImpl_->getIfExists("status/@id", status) )
            throw FatalError( NSessionCodes::UNKNOWN,
                              "Bad blackbox response: missing status/@id"
                  );
        status_ = static_cast<Status>(status);

        // empty if no <age> found
        pImpl_->getIfExists("age", age_);

        long int uid;
        isLite_ = pImpl_->getIfExists("liteuid", uid);
    }

    SessionResp::~SessionResp() {
    }

    MultiSessionResp::MultiSessionResp(auto_ptr<Impl> pImpl)
        : Response(pImpl)
    {
        long int status;
        if ( !pImpl_->getIfExists("status/@id", status) )
            throw FatalError( NSessionCodes::UNKNOWN,
                              "Bad blackbox response: missing status/@id"
                  );
        status_ = static_cast<SessionResp::Status>(status);

        // empty if no <age> found
        pImpl_->getIfExists("age", age_);

        pImpl_->getIfExists("default_uid", default_uid_);

        xmlConfig::Parts users = pImpl_->getParts("user");
        count_ = users.Size();
    }

    MultiSessionResp::~MultiSessionResp() {
    }

    string MultiSessionResp::id(int i) const {
        if ( i<0 || i>=count_ )
            throw FatalError( NSessionCodes::INVALID_PARAMS,
                              "Multisession response index out of range" );

        xmlConfig::Part user = pImpl_->getParts("user")[i];

        string id;
        user.GetIfExists("@id", id);

        return id;
    }

    auto_ptr<SessionResp> MultiSessionResp::user(int i) const {
        if ( i<0 || i>=count_ )
            throw FatalError( NSessionCodes::INVALID_PARAMS,
                              "Multisession response index out of range" );

        xmlConfig::Part user = pImpl_->getParts("user")[i];

        SessionResp* pResp = new SessionResp( auto_ptr<Response::Impl>(
            new Response::Impl(&user) ) );

        return auto_ptr<SessionResp> (pResp);
    }

    LoginResp::LoginResp(auto_ptr<Impl> pImpl)
        : Response(pImpl), accStatus_(accUnknown), pwdStatus_(pwdUnknown)
    {
        long int status;
        if ( !pImpl_->getIfExists("status/@id", status) )
        {
            // Try login v2 fields
            long int astatus, pstatus;

            if (!pImpl_->getIfExists("login_status/@id", astatus) ||
                !pImpl_->getIfExists("password_status/@id", pstatus) )
                throw FatalError( NSessionCodes::UNKNOWN,
                                "Bad blackbox response: missing login status"
                    );
            accStatus_ = static_cast<AccountStatus>(astatus);
            pwdStatus_ = static_cast<PasswdStatus>(pstatus);

            // set up legacy v1 status field
            string tmp;
            if (accStatus_==accValid && pwdStatus_==pwdValid)
                status_ = Valid;
            else if (accStatus_==accDisabled)
                status_ = Disabled;
            else if (accStatus_==accAlienDomain)
                status_ = AlienDomain;
            else if (pwdStatus_==pwdCompromised)
                status_ = Compromised;
            else if (pImpl_->getIfExists("bruteforce_policy/captcha", tmp))
                status_ = ShowCaptcha;
            else if (pImpl_->getIfExists("bruteforce_policy/password_expired", tmp))
                status_ = Expired;
            else status_ = Invalid;
        }
        else status_ = static_cast<Status>(status);

        string comment;
        pImpl_->getIfExists("comment", comment);
        if (!comment.empty())
            message_ = comment;
    }

    LoginResp::~LoginResp() {
    }

    HostResp::HostResp(auto_ptr<Response::Impl> pImpl)
        : pImpl_(pImpl), message_()
    {
        if ( !pImpl_.get() )
            throw FatalError( NSessionCodes::UNKNOWN,
                              "Internal error: NULL Response::Impl" );

        long int status;
        if ( !pImpl_->getIfExists("status/@id", status) )
            throw FatalError( NSessionCodes::UNKNOWN,
                              "Bad blackbox response: missing status/@id"
                  );
        status_ = static_cast<Status>(status);

        pImpl_->getIfExists("status", message_);
    }

    HostResp::~HostResp() {
    }

    PwdQualityResp::PwdQualityResp(std::auto_ptr<Response::Impl> pImpl)
        : pImpl_(pImpl), message_()
    {
        if ( !pImpl_.get() )
            throw FatalError( NSessionCodes::UNKNOWN,
                              "Internal error: NULL Response::Impl" );

        // empty if no <error> tag
        pImpl_->getIfExists("error", message_);

        long int err_code;
        if ( pImpl_->getIfExists("exception/@id", err_code) )
        {
            NSessionCodes::ESessionError code = static_cast<NSessionCodes::ESessionError>(err_code);
            if ( code == NSessionCodes::DB_EXCEPTION ||
                 code == NSessionCodes::DB_FETCHFAILED)
                 throw TempError( code, message_);
            else throw FatalError( code, message_ );
        }
    }

    PwdQualityResp::~PwdQualityResp() {
    }

    // ==== Blackbox interface methods ====================================

    // method = userinfo
//     auto_ptr<Response> Info (const string& uid,
//         const string& userIp, const Options& options)
//     {
//         string req = InfoRequest(uid, userIp, options);
//
//         string resp;
//         Server::instance().Request(req, resp);
//
//         return InfoResponse(resp);
//     }
//
//
//     auto_ptr<Response> Info (const LoginSid& loginsid,
//         const string& userIp, const Options& options)
//     {
//         string req = InfoRequest(loginsid, userIp, options);
//
//         string resp;
//         Server::instance().Request(req, resp);
//
//         return InfoResponse(resp);
//     }
//
//
//     auto_ptr<Response> Info (const SuidSid& suidsid,
//         const string& userIp, const Options& options)
//     {
//         string req = InfoRequest(suidsid, userIp, options);
//
//         string resp;
//         Server::instance().Request(req, resp);
//
//         return InfoResponse(resp);
//     }
//
//
//     // method = login
//     auto_ptr<LoginResp> Login (const LoginSid& loginsid,
//         const string& password, const string& from,
//         const string& userIp, const Options& options)
//     {
//         LoginReqData req = LoginRequest(loginsid, password, from,
//                                        userIp, options);
//
//         string resp;
//         Server::instance().Request(req.uri_, req.postData_, resp);
//
//         return LoginResponse(resp);
//     }
//
//
//     // method = sessionid
//     auto_ptr<SessionResp> SessionID (const string& sessionId,
//         const string& hostname,
//         const string& userIp, const Options& options)
//     {
//         string req = SessionIDRequest(sessionId, hostname, userIp, options);
//
//         string resp;
//         Server::instance().Request(req, resp);
//
//         return SessionIDResponse(resp);
//     }

    // ==== Common constants ==============================================

    const string strMethodlogin         ("method=login");
    const string strMethodSessionid     ("method=sessionid");
    const string strMethodOAuth         ("method=oauth");
    const string strMethodUserinfo      ("method=userinfo");
    const string strMethodMailhost      ("method=mailhost");
    const string strMethodPwdQuality    ("method=pwdquality");

    // ==== Blackbox interface without network operations =================

    static void appendSign(std::string& req, const ConsumerContext& signCtx, const std::string& meta, const std::string& concat) {
        req.append("&consumer=").append(signCtx.consumer());
        req.append("&client_id=").append(signCtx.clientId());

        static const size_t MAX_TS_LEN = 20; // std::numeric_limits<time_t>::max() == 9223372036854775807
        std::string ts(MAX_TS_LEN, 0x00);
        std::snprintf(const_cast<char*>(ts.data()), MAX_TS_LEN, "%ld", std::time(0));
        ts.resize(ts.find('\0'));
        req.append("&ts=").append(ts);

        std::string metaB64 = bin2base64url(meta);
        if (!metaB64.empty()) {
            req.append("&meta_b64u=").append(metaB64);
        }

        std::string toSign;
        toSign.append(ts).append(metaB64).append(concat);
        req.append("&env_sign=").append(bin2base64url(HMAC_sha256(signCtx.secret(), toSign)));
    }

    // method = userinfo
    string InfoRequest(const string& uid,
        const string& userIp, const Options& options)
    {
        RequestBuilder req(strMethodUserinfo);
        req.add("uid", uid);
        req.add("userip", userIp);
        req.add(options);

        return req;
    }

    string InfoRequestSigned(const string& uid, const string& userIp, const ConsumerContext& signCtx, const Options& options, const string& meta)
    {
        std::string req = InfoRequest(uid, userIp, options);
        appendSign(req, signCtx, meta, uid);
        return req;
    }


    string InfoRequest(const LoginSid& loginsid,
        const string& userIp, const Options& options)
    {
        RequestBuilder req(strMethodUserinfo);
        string punycode;
        const string& ref = idn::encodeAddr(loginsid.login, punycode);
        req.add("login", ref);
        if( !loginsid.sid.empty() )
            req.add("sid", loginsid.sid);
        req.add("userip", userIp);
        req.add(options);

        return req;
    }

    string InfoRequestSigned(const LoginSid& loginsid, const string& userIp, const ConsumerContext& signCtx, const Options& options, const string& meta)
    {
        std::string req = InfoRequest(loginsid, userIp, options);
        appendSign(req, signCtx, meta, loginsid.login + loginsid.sid);
        return req;
    }


    string InfoRequest(const SuidSid& suidsid,
        const string& userIp, const Options& options)
    {
        RequestBuilder req(strMethodUserinfo);
        req.add("suid", suidsid.suid);
        req.add("sid", suidsid.sid);
        req.add("userip", userIp);
        req.add(options);

        return req;
    }

    string InfoRequestSigned(const SuidSid& suidsid, const string& userIp, const ConsumerContext& signCtx, const Options& options, const string& meta)
    {
        std::string req = InfoRequest(suidsid, userIp, options);
        appendSign(req, signCtx, meta, suidsid.suid);
        return req;
    }

    string InfoRequest(const vector<string>& uids,
        const string& userIp,
        const Options& options)
    {
        RequestBuilder req(strMethodUserinfo);
        req.add("uid", uids);
        req.add("userip", userIp);
        req.add(options);

        return req;
    }

    string InfoRequestSigned(const std::vector<string>& uids, const string& userIp, const ConsumerContext& signCtx, const Options& options, const string& meta)
    {
        std::string req = InfoRequest(uids, userIp, options);
        std::string concat;
        for (size_t idx = 0; idx < uids.size(); ++idx) {
            if (!concat.empty()) {
                concat.push_back(',');
            }
            concat.append(uids[idx]);
        }
        appendSign(req, signCtx, meta, concat);
        return req;
    }

    auto_ptr<Response> InfoResponse(const string& bbResponse)
    {
        Response* pResp = new Response( auto_ptr<Response::Impl>(
            new Response::Impl(bbResponse) ) );

        return auto_ptr<Response>( pResp );
    }

    auto_ptr<BulkResponse> InfoResponseBulk(const string& bbResponse)
    {
        BulkResponse* pResp = new BulkResponse( auto_ptr<Response::Impl>(
            new Response::Impl(bbResponse) ) );

        return auto_ptr<BulkResponse>( pResp );
    }

    // method = login
    LoginReqData LoginRequest(const LoginSid& loginsid,
        const string& password, const string& authtype,
        const string& userIp, const Options& options)
    {
        LoginReqData reqData;

        RequestBuilder req(strMethodlogin);
        string punycode;
        const string& ref = idn::encodeAddr(loginsid.login, punycode);
        req.add("login", ref);
        if( !loginsid.sid.empty() )
            req.add("sid", loginsid.sid);
        req.add("authtype", authtype);
        req.add("userip", userIp);
        req.add(options);

        reqData.uri_ = req;

        reqData.postData_.reserve(password.size() * 2);
        reqData.postData_.append("password=").append( URLEncode(password) );

        return reqData;
    }

    LoginReqData LoginRequestSigned(const LoginSid& loginsid, const string& password, const string& authtype,
                              const string& userIp, const ConsumerContext& signCtx, const Options& options, const string& meta)
    {
        LoginReqData req = LoginRequest(loginsid, password, authtype, userIp, options);
        std::string concat;
        concat.append(userIp).append(password).append(loginsid.login).append(loginsid.sid).append(authtype);
        appendSign(req.uri_, signCtx, meta, concat);
        return req;
    }

    LoginReqData LoginRequestUid(const std::string& uid,
        const string& password, const string& authtype,
        const string& userIp, const Options& options)
    {
        LoginReqData reqData;

        RequestBuilder req(strMethodlogin);
        req.add("uid", uid);
        req.add("authtype", authtype);
        req.add("userip", userIp);
        req.add(options);

        reqData.uri_ = req;

        reqData.postData_.reserve(password.size() * 2);
        reqData.postData_.append("password=").append( URLEncode(password) );

        return reqData;
    }

    LoginReqData LoginRequestUidSigned(const string& uid, const string& password, const string& authtype, const string& userIp, const ConsumerContext& signCtx, const Options& options, const string& meta)
    {
        LoginReqData req = LoginRequest(uid, password, authtype, userIp, options);
        std::string concat;
        concat.append(userIp).append(password).append(uid).append(authtype);
        appendSign(req.uri_, signCtx, meta, concat);
        return req;
    }

    auto_ptr<LoginResp> LoginResponse(const string& bbResponse)
    {
        LoginResp* pResp = new LoginResp( auto_ptr<Response::Impl>(
            new Response::Impl(bbResponse) ) );

        return auto_ptr<LoginResp>( pResp );
    }


    // method = sessionid
    string SessionIDRequest(const string& sessionId,
        const string& hostname,
        const string& userIp, const Options& options)
    {
        RequestBuilder req(strMethodSessionid);
        req.add("sessionid", sessionId);
        req.add("host", hostname);
        req.add("userip", userIp);
        req.add(options);

        return req;;
    }

    string SessionIDRequestSigned(const string& sessionId, const string& hostname, const string& userIp,
                                  const ConsumerContext& signCtx, const Options& options, const string& meta)
    {
        std::string req = SessionIDRequest(sessionId, hostname, userIp, options);
        std::string concat;
        concat.append(userIp).append(hostname).append(sessionId);
        appendSign(req, signCtx, meta, concat);
        return req;
    }

    auto_ptr<SessionResp> SessionIDResponse(const string& bbResponse)
    {
        SessionResp* pResp = new SessionResp( auto_ptr<Response::Impl>(
            new Response::Impl(bbResponse) ) );

        return auto_ptr<SessionResp>( pResp );
    }

    auto_ptr<MultiSessionResp> SessionIDResponseMulti(const string& bbResponse)
    {
        MultiSessionResp* pResp = new MultiSessionResp( auto_ptr<Response::Impl>(
            new Response::Impl(bbResponse) ) );

        return auto_ptr<MultiSessionResp>( pResp );
    }

    // method = oauth

    LoginReqData OAuthRequestSecure(const string& token,
        const string& userIp,
        const Options& options)
    {
        RequestBuilder req(strMethodOAuth);
        req.add("userip", userIp);
        req.add(options);

        LoginReqData loginData;
        loginData.uri_ = req;
        loginData.postData_.append("oauth_token=").append(URLEncode(token));

        return loginData;
    }

    LoginReqData OAuthRequestSignedSecure(const string& token,
        const string& userIp,
        const ConsumerContext& signCtx,
        const Options& options,
        const string& meta)
    {
        LoginReqData oAuthReq = OAuthRequestSecure(token, userIp, options);

        std::string concat;
        concat.append(userIp).append(token);

        for (size_t idx = 0; idx < options.list().size(); ++idx) {
            if (options.list()[idx].key() == "scopes" ) {
                concat.append(options.list()[idx].val());
                break;
            }
        }

        appendSign(oAuthReq.uri_, signCtx, meta, concat);
        return oAuthReq;
    }

    // method = mailhost
    string MailHostCreateRequest(
        const string& scope,
        const string& dbId,
        const string& priority,
        const string& mx)
    {
        RequestBuilder req(strMethodMailhost);
        req.add("op", "create");
        req.add("scope", scope);
        req.add("db_id", dbId);
        req.add("prio", priority);
        req.add("mx", mx);
        return req;
    }

    string MailHostDeleteRequest(
        const string& scope,
        const string& dbId)
    {
        RequestBuilder req(strMethodMailhost);
        req.add("op", "delete");
        req.add("scope", scope);
        req.add("db_id", dbId);

        return req;
    }

    string MailHostAssignRequest(
        const string& scope,
        const string& suid,
        const string& dbId,
        const string& oldDbId)
    {
        RequestBuilder req(strMethodMailhost);
        req.add("op", "assign");
        req.add("scope", scope);
        req.add("suid", suid);
        req.add("db_id", dbId);

        if (!oldDbId.empty())
            req.add("old_db_id", oldDbId);

        return req;
    }

    string MailHostSetPriorityRequest(
        const string& scope,
        const string& dbId,
        const string& priority)
    {
        RequestBuilder req(strMethodMailhost);
        req.add("op", "setprio");
        req.add("scope", scope);
        req.add("db_id", dbId);
        req.add("prio", priority);

        return req;
    }

    string MailHostFindRequest(
        const string& scope,
        const string& priority)
    {
        RequestBuilder req(strMethodMailhost);
        req.add("op", "find");
        req.add("scope", scope);
        req.add("prio", priority);

        return req;
    }

    auto_ptr<HostResp> MailHostResponse(const string& bbResponse)
    {
        HostResp* pResp = new HostResp( auto_ptr<Response::Impl>(
            new Response::Impl(bbResponse) ) );

        return auto_ptr<HostResp>( pResp );
    }

    string PwdQualityRequest(
        const string& sessionId,
        const string& hostname)
    {
        RequestBuilder req(strMethodPwdQuality);
        req.add("sessionid", sessionId);
        req.add("host", hostname);

        return req;
    }

    auto_ptr<PwdQualityResp> PwdQualityResponse (const string& bbResponse) {
        PwdQualityResp* pResp = new PwdQualityResp( auto_ptr<Response::Impl>(
            new Response::Impl(bbResponse) ) );

        return auto_ptr<PwdQualityResp>( pResp );
    }

}   // namespace bb

// vi: expandtab:sw=4:ts=4
// kate: replace-tabs on; indent-width 4; tab-width 4;
