// ====================================================================
//  Implementation of blackbox response options
// ====================================================================

#include "yandex/blackbox/options.h"

#include "utils.h"
#include "responseimpl.h"

namespace bb {
    using namespace std;

    /// Construct options list with one option inside
    Options::Options (const Option& opt)
        : options_(1, opt)
    {}

    /// Add option to the list as stream
    Options& Options::operator<< (const Option& opt)
    {
        options_.push_back(opt);
        return *this;
    }

    /// Transform email option to options list
    Options& OptTestEmail::format (Options& opts) const
    {
        opts << Option("emails", "testone");
        string punycode;
        const string& ref = idn::encodeAddr(addrToTest_, punycode);
        opts << Option("addrtotest", ref);

        return opts;
    }

    DBFields::DBFields (const string& field)
    {
        // add one field key
        if ( !field.empty() ) fields_[field];
    }

    DBFields::DBFields (const Response* pResp)
    {
        readResponse(pResp);
    }

    void DBFields::readResponse (const Response* pResp)
    {
        if ( !pResp )
            throw FatalError(NSessionCodes::INVALID_PARAMS, "NULL Response pointer");

        // need to clean old values first
        clearValues();

        // Then add all fields coming from Response
        Response::Impl* pImpl = pResp->getImpl();

        xmlConfig::Parts fields ( pImpl->getParts("dbfield") );
        string id, val;

        for (int i=0; i < fields.Size(); ++i)
        {
            xmlConfig::Part item(fields[i]);
            if ( !item.GetIfExists("@id", id) )
                throw FatalError(NSessionCodes::UNKNOWN,
                                 "Bad blackbox response: missing dbfield/@id");

            if ( id.find("login") != id.npos ||
                 id.find("email") != id.npos)
            {
                string tmp;
                string itemString = item.asString();
                const string& ref = idn::decodeAddr(itemString, tmp);
                val = ref;
            }
            else val = item.asString();

            fields_[id]=val;
        }
    }

    void DBFields::clear()
    {
        fields_.clear();
    }

    void DBFields::clearValues()
    {
        container_type::iterator p = fields_.begin();

        while( p != fields_.end() )
            p++->second.clear();
    }

    DBFields& DBFields::operator<< (const string& field)
    {
        // adds the key if does not exist
        if ( !field.empty() ) fields_[field];
        return *this;
    }

    const string& DBFields::get (const string& field) const
    {
        const_iterator it = fields_.find(field);

        if( it != fields_.end() )
            return it->second;
        else
            return emptyStr;
    }

    bool DBFields::has(const string& field) const
    {
        return end() != fields_.find(field);
    }

    DBFields::const_iterator DBFields::begin() const
    {
        return fields_.begin();
    }

    DBFields::const_iterator DBFields::end() const
    {
        return fields_.end();
    }

    DBFields::operator Options() const
    {
        Options r;
        return r << *this;
    }

    Options& operator<< (Options& options, const DBFields& opt)
    {
        string fields_list;

        if( opt.empty() ) return options;   // don't write if empty

        DBFields::const_iterator it = opt.begin();

        fields_list.append(it++->first);

        while( it != opt.end() )
        {
            fields_list.append(1, ',');
            fields_list.append(it++->first);
        }

        return options << Option("dbfields", fields_list);
    }

    Options& operator<< (Options& options, const OptTestEmail& opt)
    { return opt.format(options); }


    OptAliases::OptAliases (const string& alias)
    {
        if (!alias.empty()) aliases_.insert(alias);
    }

    OptAliases& OptAliases::operator<< (const string& alias)
    {
        if (!alias.empty()) aliases_.insert(alias);
        return *this;
    }

    OptAliases::operator Options() const
    {
        Options r;
        return r << *this;
    }

    Option OptAliases::format() const
    {
        string aliases_list;

        set<string>::const_iterator it = aliases_.begin();

        if (it != aliases_.end())
            aliases_list.append(*it++);

        while( it != aliases_.end() )
        {
            aliases_list.append(1, ',');
            aliases_list.append(*it++);
        }

        return Option("aliases", aliases_list);
    }

    Options& operator<< (Options& options, const OptAliases& opt)
    {
        if (!opt.empty())
            options << opt.format();
        return options;
    }

    Attributes::Attributes (const string& attr)
    {
        // add one field key
        if ( !attr.empty() ) attributes_[attr];
    }

    Attributes::Attributes (const Response* pResp)
    {
        readResponse(pResp);
    }

    void Attributes::readResponse (const Response* pResp)
    {
        if ( !pResp )
            throw FatalError(NSessionCodes::INVALID_PARAMS, "NULL Response pointer");

        // need to clean old values first
        clearValues();

        // Then add all attrs coming from Response
        Response::Impl* pImpl = pResp->getImpl();

        xmlConfig::Parts attrs ( pImpl->getParts("attributes/attribute") );
        string type, val;

        for (int i=0; i < attrs.Size(); ++i)
        {
            xmlConfig::Part item(attrs[i]);
            if ( !item.GetIfExists("@type", type) )
                throw FatalError(NSessionCodes::UNKNOWN,
                                 "Bad blackbox response: missing attribute/@type");

            val = item.asString();

            attributes_[type]=val;
        }
    }

    void Attributes::clear()
    {
        attributes_.clear();
    }

    void Attributes::clearValues()
    {
        container_type::iterator p = attributes_.begin();

        while( p != attributes_.end() )
            p++->second.clear();
    }

    Attributes& Attributes::operator<< (const string& attr)
    {
        // adds the key if does not exist
        if ( !attr.empty() ) attributes_[attr];
        return *this;
    }

    const string& Attributes::get (const string& attr) const
    {
        const_iterator it = attributes_.find(attr);

        if( it != attributes_.end() )
            return it->second;
        else
            return emptyStr;
    }

    bool Attributes::has(const string& attr) const
    {
        return end() != attributes_.find(attr);
    }

    Attributes::const_iterator Attributes::begin() const
    {
        return attributes_.begin();
    }

    Attributes::const_iterator Attributes::end() const
    {
        return attributes_.end();
    }

    Attributes::operator Options() const
    {
        Options r;
        return r << *this;
    }

    Option Attributes::format() const
    {
        string attributes_list;

        Attributes::const_iterator it = attributes_.begin();

        if (it != attributes_.end())
            attributes_list.append(it++->first);

        while( it != attributes_.end() )
        {
            attributes_list.append(1, ',');
            attributes_list.append(it++->first);
        }

        return Option("attributes", attributes_list);
    }

    Options& operator<< (Options& options, const Attributes& attrs)
    {
        if (!attrs.empty())
            options << attrs.format();
        return options;
    }

    // ==== Static constants ==============================================

    const string emptyStr;

    const Options optNone;

    const Option optRegname             ("regname",     "yes");
    const Option optAuthId              ("authid",      "yes");
    const Option optFullInfo            ("full_info",   "yes");

    const Option optVersion2            ("ver",         "2");
    const Option optMultisession        ("multisession","yes");

    const Option optGetSocialAliases    ("aliases",     "getsocial");
    const Option optGetAllAliases       ("aliases",     "all");

    const Option optGetAllEmails        ("emails",      "getall");
    const Option optGetYandexEmails     ("emails",      "getyandex");
    const Option optGetDefaultEmail     ("emails",      "getdefault");

    const Option optGetUserTicket       ("get_user_ticket", "yes");

}    // namespace bb

// vi: expandtab:sw=4:ts=4
// kate: replace-tabs on; indent-width 4; tab-width 4;
