// ====================================================================
//  Miscelaneous library utilities and helpers
// ====================================================================

#include <arpa/inet.h>
#include <stdexcept>

#include <lib/idna.h>

#include "yandex/blackbox/blackbox2.h"

#include "utils.h"
//#include "curlwrapper.h"

#include <openssl/evp.h>
#include <openssl/hmac.h>
#include <openssl/sha.h>

namespace bb
{
    /// convert hex character to int value
    int hex2int(char c)
    {
        if ( '0' <= c && c <= '9' ) return c - '0';
        if ( 'a' <= c && c <= 'f' ) return c - 'a' + 10;
        if ( 'A' <= c && c <= 'F' ) return c - 'A' + 10;

        return -1;      // error
    }

    /// convert int value 0..15 to hex char
    char int2hex(int n)
    {
        if ( 0 <= n && n <= 9 )  return static_cast<char>('0' + n);
        if ( 10 <= n && n <=15 ) return static_cast<char>('A' + n - 10);

        return ' ';     // error
    }

    const string illegal = "!*'();:@&=+$./?#[]%";

    string URLEncode(const string& str)
    {
        const size_t len = str.size();

        string res;
        res.reserve(len);
        char hex[3] = {'%', ' ', ' '};

        for(size_t i = 0; i < len; ++i)
        {
            char c = str[i];
            if ( !isprint(c) || isspace(c) || illegal.find(c) != string::npos )
            {
                hex[1] = int2hex( (c >> 4) & 0xF );
                hex[2] = int2hex( c & 0xF);
                res.append(hex, 3);
            }
            else res.append(1, c);
        }

        return res;

        //return CUrl::get().URLEncode(str);
    }

    string URLDecode(const string& str)
    {
        const size_t len = str.size();

        string res;
        res.reserve(len);

        size_t p, q;
        p = 0;

        do {
            q = str.find('%', p);
            if ( q == string::npos )
            {
                res.append(str, p, len-p );
                return res;
            }
            if ( len-q < 2 ) // no 2 letters after %
                return "";   // error failed to parse

            res.append(str, p, q-p);

            ++q;    // skip %
            int c1 = hex2int( str[q] );
            ++q;
            int c2 = hex2int( str[q] );
            ++q;

            if ( c1 >= 0 && c2 >= 0 ) res.append(1, static_cast<char>((c1 << 4) + c2) );
                else return "";     // failed to parse

            p = q;

        } while ( p < len );

        return res;

        //return CUrl::get().URLDecode(str);
    }


    namespace {
        static const unsigned char b64_encode[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_";
        static unsigned char b64_decode[256];
        static bool b64_init() {
        for (int i = 0; i < 256; ++i)
            b64_decode[i] = 0xff;

        for (int i = 0; i < 64; ++i)
            b64_decode[ b64_encode[i] ] = static_cast<unsigned char>(i);

        return true;
    }

    static bool b64_decode_inited = b64_init();
    }

    std::string bin2base64url(const char* buf, size_t len, bool pad) {
        if (buf == NULL) {
            return std::string();
        }

        std::string res;
        res.resize( ((len+2)/3) << 2, 0 );

        const unsigned char* pB = reinterpret_cast<const unsigned char*>(buf);
        const unsigned char* pE = reinterpret_cast<const unsigned char*>(buf) + len;
        unsigned char* p = reinterpret_cast<unsigned char*>(const_cast<char*>(res.data()));
        for ( ; pB + 2 < pE; pB += 3) {
            const unsigned char a = *pB;
            *p++ = b64_encode[(a >> 2) & 0x3F];
            const unsigned char b = *(pB + 1);
            *p++ = b64_encode[((a & 0x3) << 4) | ((b & 0xF0) >> 4)];
            const unsigned char c = *(pB + 2);
            *p++ = b64_encode[((b & 0xF) << 2) | ((c & 0xC0) >> 6)];
            *p++ = b64_encode[c & 0x3F];
        }

        if (pB < pE) {
            const unsigned char a = *pB;
            *p++ = b64_encode[(a >> 2) & 0x3F];
            if (pB == (pE - 1)) {
                *p++ = b64_encode[((a & 0x3) << 4)];
                if (pad) {
                    *p++ = '=';
                }
            }
            else {
                const unsigned char b = *(pB + 1);
                *p++ = b64_encode[((a & 0x3) << 4) |
                                (static_cast<int> (b & 0xF0) >> 4)];
                *p++ = b64_encode[((b & 0xF) << 2)];
            }
            if (pad) {
                *p++ = '=';
            }
        }

        res.resize(static_cast<size_t>(p - reinterpret_cast<const unsigned char*>(res.data())));
        return res;
    }

    std::string base64url2bin(const char* buf, size_t len) {

        const unsigned char *bufin = reinterpret_cast<const unsigned char*>(buf);
        if (bufin == NULL || len == 0 || b64_decode[*bufin] > 63) {
            return std::string();
        }
        const unsigned char* bufend = reinterpret_cast<const unsigned char*>(buf) + len;
        while (++bufin < bufend && b64_decode[*bufin] < 64);
        int nprbytes = static_cast<int>(bufin - reinterpret_cast<const unsigned char *>(buf));
        size_t nbytesdecoded = static_cast<size_t>((nprbytes + 3) / 4) * 3;

        if (nprbytes < static_cast<int>(len)) {
            int left = static_cast<int>(len) - nprbytes;
            while (left--) {
                if (*(bufin++) != '=') return std::string();
            }
        }

        std::string res;
        res.resize(nbytesdecoded);

        unsigned char *bufout = reinterpret_cast<unsigned char *>(const_cast<char*>(res.data()));
        bufin = reinterpret_cast<const unsigned char *>(buf);

        while (nprbytes > 4) {
            unsigned char a = b64_decode[*bufin];
            unsigned char b = b64_decode[bufin[1]];
            *(bufout++) = static_cast<unsigned char> (a << 2 | b >> 4);
            unsigned char c = b64_decode[bufin[2]];
            *(bufout++) = static_cast<unsigned char> (b << 4 | c >> 2);
            unsigned char d = b64_decode[bufin[3]];
            *(bufout++) = static_cast<unsigned char> (c << 6 | d);
            bufin += 4;
            nprbytes -= 4;
        }

        /* Note: (nprbytes == 1) would be an error, so just ingore that case */
        if (nprbytes > 1) {
            *(bufout++) = static_cast<unsigned char> (b64_decode[*bufin] << 2 | b64_decode[bufin[1]] >> 4);
        }
        if (nprbytes > 2) {
            *(bufout++) = static_cast<unsigned char> (b64_decode[bufin[1]] << 4 | b64_decode[bufin[2]] >> 2);
        }
        if (nprbytes > 3) {
            *(bufout++) = static_cast<unsigned char> (b64_decode[bufin[2]] << 6 | b64_decode[bufin[3]]);
        }

        int diff = (4 - nprbytes) & 3;
        if (diff) {
            nbytesdecoded -= (4 - nprbytes) & 3;
            res.resize(nbytesdecoded);
        }

        return res;
    }

    std::string bin2base64url(const std::string& buf, bool pad) {
        return bin2base64url(buf.data(), buf.size(), pad);
    }

    std::string base64url2bin(const std::string& buf) {
        return base64url2bin(buf.data(), buf.size());
    }

    std::string HMAC_sha256(const std::string& key, const char* data, size_t dataLen) {
        std::string value(EVP_MAX_MD_SIZE, 0);
        unsigned macLen = 0;

        unsigned char* res = ::HMAC(EVP_sha256(), key.data(), static_cast<int>(key.size()),
                                    reinterpret_cast<unsigned char*>(const_cast<char*>(data)),
                                    dataLen, reinterpret_cast<unsigned char*>(const_cast<char*>(value.data())),
                                    &macLen);

        if (!res)
        {
            return std::string();
        }

        if (macLen != EVP_MAX_MD_SIZE) {
            value.resize(macLen);
        }
        return value;
    }

    std::string HMAC_sha256(const std::string& key, const std::string& data) {
        return HMAC_sha256(key, data.data(), data.size());
    }

    RequestBuilder::RequestBuilder(const string& method)
    {
        request_.reserve(MAX_REQUEST_LENGTH);
        request_.assign(method);    // don't escape method
    }

    void RequestBuilder::add(const string& key, const string& val)
    {
        request_.append(1, '&').append(URLEncode(key)).append(1, '=').append(URLEncode(val));
    }

    void RequestBuilder::add(const string& key, const vector<string>& values)
    {
        string val;

        vector<string>::const_iterator p = values.begin();

        if( p!=values.end() )
            val.append(*p++);

        while( p!=values.end() ) {
            val.append(1, ',');
            val.append(*p++);
        }

        add(key, val);
    }

    void RequestBuilder::add(const Option& opt)
    {
        add(opt.key(), opt.val());
    }

    void RequestBuilder::add(const Options& options)
    {
        const Options::ListType& list = options.list();

        Options::ListType::const_iterator it = list.begin();

        while( it != list.end() )
        {
            add( it->key(), it->val() );
            ++it;
        }
    }

    ostream& operator<< (ostream& out, LoginResp::Status s)
    {
        switch(s) {
            case LoginResp::Valid:          out << "Valid";         break;
            case LoginResp::Disabled:       out << "Disabled";      break;
            case LoginResp::Invalid:        out << "Invalid";       break;
            case LoginResp::ShowCaptcha:    out << "ShowCaptcha";   break;
            case LoginResp::AlienDomain:    out << "AlienDomain";   break;
            case LoginResp::Compromised:    out << "Compromised";   break;
            case LoginResp::Expired:        out << "Expired";       break;
            default:                        out << "<UNKNOWN LOGIN STATUS>";
        }
        return out;
    }

    ostream& operator<< (ostream& out, LoginResp::AccountStatus s)
    {
        switch(s) {
            case LoginResp::accUnknown:     out << "Unknown";       break;
            case LoginResp::accValid:       out << "Valid";         break;
            case LoginResp::accAlienDomain: out << "AlienDomain";   break;
            case LoginResp::accNotFound:    out << "NotFound";      break;
            case LoginResp::accDisabled:    out << "Disabled";      break;
            default:                        out << "<UNKNOWN ACCOUNT STATUS>";
        }
        return out;
    }

    ostream& operator<< (ostream& out, LoginResp::PasswdStatus s)
    {
        switch(s) {
            case LoginResp::pwdUnknown:     out << "Unknown";       break;
            case LoginResp::pwdValid:       out << "Valid";         break;
            case LoginResp::pwdBad:         out << "Bad";           break;
            case LoginResp::pwdCompromised: out << "Compromised";   break;
            default:                        out << "<UNKNOWN PASSWORD STATUS>";
        }
        return out;
    }

    ostream& operator<< (ostream& out, SessionResp::Status s)
    {
        switch(s) {
            case SessionResp::Valid:        out << "Valid";     break;
            case SessionResp::NeedReset:    out << "NeedReset"; break;
            case SessionResp::Expired:      out << "Expired";   break;
            case SessionResp::NoAuth:       out << "NoAuth";    break;
            case SessionResp::Disabled:     out << "Disabled";  break;
            case SessionResp::Invalid:      out << "Invalid";   break;
            case SessionResp::WrongGuard:   out << "WrongGuard";break;
            default:                        out << "<UNKNOWN SESSION STATUS>";
        }
        return out;
    }

    ostream& operator<< (ostream& out, HostResp::Status s)
    {
        switch(s) {
            case HostResp::OK:              out << "OK";            break;
            case HostResp::AccessDenied:    out << "AccessDenied";  break;
            case HostResp::MissingParam:    out << "MissingParam";  break;
            case HostResp::UnknownOp:       out << "UnknownOp";     break;
            case HostResp::UnknownScope:    out << "UnknownScope";  break;
            case HostResp::InvalidParam:    out << "InvalidParam";  break;
            case HostResp::RecordNotFound:  out << "RecordNotFound";break;
            default:                        out << "<UNKNOWN HOST STATUS>";
        }
        return out;
    }

    ostream& operator<< (ostream& out, AliasList::Item::Type t)
    {
        switch(t) {
            case AliasList::Item::Portal:       out << "Portal";    break;
            case AliasList::Item::Mail:         out << "Mail";      break;
            case AliasList::Item::NarodMail:    out << "NarodMail"; break;
            case AliasList::Item::Lite:         out << "Lite";      break;
            case AliasList::Item::Social:       out << "Social";    break;
            case AliasList::Item::PDD:          out << "PDD";       break;
            case AliasList::Item::PDDAlias:     out << "PDDAlias";  break;
            case AliasList::Item::AltDomain:    out << "AltDomain"; break;
            case AliasList::Item::Phonish:      out << "Phonish";   break;
            case AliasList::Item::PhoneNumber:  out << "PhoneNumber"; break;
            case AliasList::Item::Mailish:      out << "Mailish";   break;
            case AliasList::Item::Yandexoid:    out << "Yandexoid"; break;
            case AliasList::Item::KinopoiskId:  out << "KinopoiskId";break;
            default:                            out << static_cast<unsigned int>(t);
        }
        return out;
    }

    ostream& operator<< (ostream& out, NSessionCodes::ESessionError e)
    {
        switch(e) {
            case NSessionCodes::OK:              out << "OK";            break;
            case NSessionCodes::UNKNOWN:         out << "Unknown";       break;
            case NSessionCodes::INVALID_PARAMS:  out << "InvalidParams"; break;
            case NSessionCodes::DB_FETCHFAILED:  out << "DBFetchFailed"; break;
            case NSessionCodes::DB_EXCEPTION:    out << "DBException";   break;
            case NSessionCodes::ACCESS_DENIED:   out << "AccessDenied";  break;
            default:                            out << static_cast<unsigned int>(e);
        }
        return out;
    }

    ostream& operator<< (ostream& out, BruteforcePolicy::Mode m)
    {
        switch(m) {
            case BruteforcePolicy::None:        out << "None";          break;
            case BruteforcePolicy::Captcha:     out << "Captcha";       break;
            case BruteforcePolicy::Delay:       out << "Delay";         break;
            case BruteforcePolicy::Expired:     out << "Expired";       break;
            default:                            out << "<Unknown>";
        }
        return out;
    }

    ostream& operator<< (ostream& out, SessionKind::Kind k)
    {
        switch(k) {
            case SessionKind::None:             out << "None";          break;
            case SessionKind::Support:          out << "Support";       break;
            case SessionKind::Stress:           out << "Stress";        break;
            default:                            out << static_cast<unsigned>(k);
        }
        return out;
    }

    ostream& operator<< (ostream& out, const EmailList::Item& item)
    {
        out << "addr='" << item.address() << "', born date='" << item.date() << "'";

        if ( item.isDefault() ) out << ", default";

        if ( item.native() ) out << ", native";
        else if ( item.rpop() ) out << ", remote pop";
        else out << (item.validated() ? ",":", not") << " validated";

        return out;
    }

    ostream& operator<< (ostream& out, const AliasList::Item& item)
    {
        out << "type='" << item.type() << "', alias='" << item.alias() << "'";

        return out;
    }

    ostream& operator<< (ostream& out, const MailHostsList::Item& item)
    {
        out << "hostId='" << item.hostId() << "', hostIp='" << item.hostIp()
            << "', hostName='" << item.hostName() << "', dbId='" << item.dbId()
            << "', sid='" << item.sid() << "', priority='" << item.priority()
            << "', mx='" << item.mx() << "'";

        return out;
    }

    ostream& operator<< (ostream& out, const pair<string,string>& item)
    {
        out << "'" << item.first << "':'" << item.second << "'";

        return out;
    }

    ostream& operator<< (ostream& out, const PDDUserInfo& info)
    {
        out << "PDD user info: domainId=" << info.domId()
            << ", domain='" << info.domain()
            << "', mx='" << info.mx()
            << "', domEna='" << info.domEna()
            << "', catchAll='" << info.catchAll() << "'";
        return out;
    }

    ostream& operator<< (ostream& out, const KarmaInfo& info)
    {
        out << "Karma info: karma=" << info.karma() << ", banTime='" << info.banTime()
            << "', karmaStatus='" << info.karmaStatus() << "'";
        return out;
    }

    ostream& operator<< (ostream& out, const DisplayNameInfo& info)
    {
        out << "DisplayName: '" << info.name() << "'";
        if ( info.social() )
            out << ", social account, provider='" << info.socProvider()
                << "', profileId=" << info.socProfile()
                << ", redirectTarget=" << info.socTarget();
        if ( !info.defaultAvatar().empty())
            out << endl << "Default avatar: " << info.defaultAvatar();
        return out;
    }

    ostream& operator<< (ostream& out, const AuthInfo& info)
    {
        out << "Password verified " << info.age() << " sec ago";
        if ( info.social() )
            out << ", social authorization, profile id=" << info.profileId();
        out << ", session " << (info.secure()?"secure":"insecure");
        return out;
    }

    ostream& operator<< (ostream& out, const EmailList& data)
    {
        if ( data.empty() ) return out;    // no emails, no default

        const EmailList::ListType& mlist = data.getEmailItems();

        EmailList::ListType::const_iterator p = mlist.begin();

        out << "Emails list: (size=" << mlist.size() << ") [" << endl;;
        while ( p != mlist.end() )
            out << "\t" << *p++ << endl;
        out << "]" << endl;

        if ( !data.getDefault().empty() )
            out << "Default email: '" << data.getDefault() << "'" << endl;

        return out;
    }

    ostream& operator<< (ostream& out, const AliasList& data)
    {
        if ( data.empty() ) return out;

        const AliasList::ListType& alist = data.getAliases();
        AliasList::ListType::const_iterator p = alist.begin();

        out << "Alias list: (size=" << alist.size() << ") [" << endl;;
        while ( p != alist.end() )
            out << "\t" << *p++ << endl;
        out << "]" << endl;

        return out;
    }

    ostream& operator<< (ostream& out, const DBFields& data)
    {
        if ( data.empty() ) return out;

        DBFields::const_iterator p = data.begin();

        out << "DBFields: (size=" << data.size() << ") [" << endl;
        while ( p != data.end() ) {
            out << "\t'" << p->first << "'='" << p->second << "'" << endl;
            ++p;
        }
        out << "]" << endl;

        return out;
    }

    ostream& operator<< (ostream& out, const Attributes& data)
    {
        if ( data.empty() ) return out;

        Attributes::const_iterator p = data.begin();

        out << "Attributes: (size=" << data.size() << ") [" << endl;
        while ( p != data.end() ) {
            out << "\t'" << p->first << "'='" << p->second << "'" << endl;
            ++p;
        }
        out << "]" << endl;

        return out;
    }

    ostream& operator<< (ostream& out, const MailHostsList& data)
    {
        if ( data.empty() ) return out;

        const MailHostsList::ListType& hlist = data.getHosts();
        MailHostsList::ListType::const_iterator p = hlist.begin();

        out << "MailHost list: (size=" << hlist.size() << ") [" << endl;
        while ( p != hlist.end() )
            out << "\t" << *p++ << endl;
        out << "]" << endl;

        return out;
    }
    ostream& operator<< (ostream& out, const OAuthInfo& data)
    {
        if ( data.empty() ) return out;

        const OAuthInfo::MapType& info = data.getInfo();
        OAuthInfo::MapType::const_iterator p = info.begin();

        out << "OAuth info: (size=" << info.size() << ") [ " << *p++;
        while ( p != info.end() )
            out << ", " << *p++;
        out << " ]" << endl;

        return out;
    }

    template <class R>
    static void PrintResponseData (ostream& out, const R* pResp)
    {
        out << "Response message: '" << pResp->message() << "'" << endl;

        // query all accessors we know
        Regname     regname(pResp);
        Uid         uid(pResp);
        LiteUid     liteUid(pResp);
        PDDUserInfo pddInfo(pResp);
        KarmaInfo   karma(pResp);
        DisplayNameInfo dispName(pResp);
        LoginInfo login(pResp);

        out << "User info: ";

        if ( !regname.value().empty() )
            out << "regname='" << regname.value() << "', ";

        if ( !uid.uid().empty() ) out << "uid="<< uid.uid();

        if ( !liteUid.liteUid().empty() )
            out << "liteUid=" << liteUid.liteUid();
        out << endl;

        if ( uid.hosted() ) out << pddInfo << endl;

        out << karma << endl;
        out << "Login name: '" << login.login() << "', havePassword="
            << (login.havePassword()?"true":"false") << ", haveHint="
            << (login.haveHint()?"true":"false") << endl;

        if ( !dispName.name().empty() ) out << dispName << endl;

        EmailList emails(pResp);
        AliasList aliases(pResp);
        DBFields fields(pResp);
        Attributes attrs(pResp);

        out << emails;
        out << aliases;
        out << fields;
        out << attrs;
    }

    ostream& operator<< (ostream& out, const Response* pResp)
    {
        if ( !pResp ) {
            out << "NULL Response." << endl;
            return out;
        }

        PrintResponseData(out, pResp);

        return out;
    }

    ostream& operator<< (ostream& out, const BulkResponse* pBulkResp)
    {
        if ( !pBulkResp ) {
            out << "NULL BulkResponse." << endl;
            return out;
        }

        out << "Bulk Response with " << pBulkResp->count() << " children"
            << " and message '" << pBulkResp->message() << "'" << endl << endl;

        for(int i=0; i<pBulkResp->count(); ++i) {
            out << "Bulk child #"<< i << " with id='" << pBulkResp->id(i) << "':" << endl;
            out << pBulkResp->user(i).get() << endl;
        }

        PrintResponseData(out, pBulkResp);

        return out;
    }

    ostream& operator<< (ostream& out, const LoginResp* pLogResp)
    {
        if ( !pLogResp ) {
            out << "NULL LoginResponse." << endl;
            return out;
        }

        out << "Login Response with status: " << pLogResp->status() << endl;
        out << "Account status: " << pLogResp->accStatus()
            << ", password status: " << pLogResp->pwdStatus() << endl;

        BruteforcePolicy policy(pLogResp);

        if ( policy.mode() != BruteforcePolicy::None ) {
            out << "Bruteforce policy: mode=" << policy.mode();
            if ( policy.mode() == BruteforcePolicy::Delay )
                out << ", delay=" << policy.delay();
            out << endl;
        }

        PasswordQuality pwdquality(pLogResp);
        ConnectionId connectionId(pLogResp);

        if ( !pwdquality.quality().empty() ) {
            out << "Password quality: " << pwdquality.quality() << endl;
        }

        if ( !connectionId.connectionId().empty() ) {
            out << "Connection Id: " << connectionId.connectionId() << endl;
        }

        PrintResponseData(out, pLogResp);

        return out;
    }

    ostream& operator<< (ostream& out, const SessionResp* pSessResp)
    {
        if ( !pSessResp ) {
            out << "NULL SessionResponse." << endl;
            return out;
        }

        out << "Session Response with status: " << pSessResp->status() << endl;

        out << "Session type: " << ( pSessResp->isLite() ? "Lite" : "Full" );
        out << ", age: " << pSessResp->age() << endl;

        NewSessionId newId(pSessResp);

        if ( !newId.value().empty() )
            out << "New sessionId='" << newId.value() << "', domain='" << newId.domain()
                << "', expires='" << newId.expires() << "', httpOnly="
                << (newId.httpOnly()?"true":"false") << endl;

        PrintResponseData(out, pSessResp);

        AuthInfo authInfo(pSessResp);
        OAuthInfo oauthInfo(pSessResp);
        SessionKind kind(pSessResp);
        AuthId authid(pSessResp);

        if ( !authInfo.age().empty() )
            out << authInfo << endl;

        if ( !oauthInfo.empty() )
            out << oauthInfo;

        if ( kind.kind() != SessionKind::None )
            out << "Session kind: " << kind.kind() << " ( '" << kind.kindName() << "' )" << endl;

        if ( !authid.authId().empty() )
            out << "AuthId: '" << authid.authId() << "', time=" << authid.time()
                << ", ip=" << authid.userIp() << ", host=" << authid.hostId() << endl;

        ConnectionId connectionId(pSessResp);

        if ( !connectionId.connectionId().empty() ) {
            out << "Connection Id: " << connectionId.connectionId() << endl;
        }

        return out;
    }

    ostream& operator<< (ostream& out, const MultiSessionResp* pMultiSessResp)
    {
        if ( !pMultiSessResp ) {
            out << "NULL MultiSessionResponse." << endl;
            return out;
        }

        out << "MultiSession Response with " << pMultiSessResp->count() << " children and status: " << pMultiSessResp->status() << endl;

        out << "Session age: " << pMultiSessResp->age() << endl;
        out << "Response message: '" << pMultiSessResp->message() << "'" << endl;
        out << "Default uid: " << pMultiSessResp->defaultUid() << endl;

        NewSessionId newId(pMultiSessResp);

        if ( !newId.value().empty() )
            out << "New sessionId='" << newId.value() << "', domain='" << newId.domain()
                << "', expires='" << newId.expires() << "', httpOnly="
                << (newId.httpOnly()?"true":"false") << endl;

        for(int i=0; i < pMultiSessResp->count(); ++i) {
            out << "Multisession child #"<< i << " with id='" << pMultiSessResp->id(i) << "':" << endl;
            out << pMultiSessResp->user(i).get() << endl;
        }

        SessionKind kind(pMultiSessResp);
        AuthId authid(pMultiSessResp);

        if ( kind.kind() != SessionKind::None )
            out << "Session kind: " << kind.kind() << " ( '" << kind.kindName() << "' )" << endl;

        if ( !authid.authId().empty() )
            out << "AuthId: '" << authid.authId() << "', time=" << authid.time()
                << ", ip=" << authid.userIp() << ", host=" << authid.hostId() << endl;

        AllowMoreUsers allow_more(pMultiSessResp);

        out << "Allow more users: " << (allow_more.allowMoreUsers()?"true":"false") << endl;

        ConnectionId connectionId(pMultiSessResp);

        if ( !connectionId.connectionId().empty() ) {
            out << "Connection Id: " << connectionId.connectionId() << endl;
        }

        return out;
    }

    ostream& operator<< (ostream& out, const HostResp* pHostResp)
    {
        if ( !pHostResp ) {
            out << "NULL Response." << endl;
            return out;
        }

        out << "Host Response with status: " << pHostResp->status()
            << " and message: '" << pHostResp->message() << "'" << endl;

        MailHostsList hosts(pHostResp);

        out << hosts;

        return out;
    }

    ostream& operator<< (ostream& out, const PwdQualityResp* pPwdQualityResp) {
        if ( !pPwdQualityResp ) {
            out << "NULL Response." << endl;
            return out;
        }

        out << "Password Quality Response with message: '"
            << pPwdQualityResp->message() << "'" << endl;

        PasswordQuality pwdquality(pPwdQualityResp);

        if ( !pwdquality.quality().empty() ) {
            out << "Password quality: " << pwdquality.quality() << endl;
        }

        return out;
    }

    /// Static Curl library initializer
    //CUrl CUrl::instance_;

}   // namespace bb

namespace idn {

    // returns true if result is in converted, false - str does not need conversion
    static bool IdnaEncode (std::string::const_iterator strBeg, std::string::const_iterator strEnd, std::string& converted)
    {
        // First, make sure original string does contain non-ASCII chars; empty
        // strings fall into this category as well
        std::string::const_iterator it;
        for (it = strBeg; it != strEnd; ++it)
            if (static_cast<unsigned char>(*it) > 127)
                break;
        if (it == strEnd)
            return false;

        // Yes, need to convert. Split into DNS parts and conver them independently
        // because each part is limited in length: it may not exceed 63 characters
        converted.clear();
        converted.reserve ( (strEnd-strBeg) * 3);

        bool firstRun = true;
        std::string part;
        std::string::const_iterator rwall = strBeg;
        for (;;) {

            // End of string
            if (rwall == strEnd)
                break;

            // Move onto the next part, which must begin with a dot in all
            // cases except for the very first one
            if (!firstRun)
                converted.push_back ('.');

            // Next part turns out to be just the dot  and nothing more
            if (++rwall == strEnd)
                break;

            // Find right wall for the current part
            std::string::const_iterator lwall = firstRun ? strBeg : rwall;
            rwall = std::find (lwall, strEnd, '.');

            // If it's empty simply skip over
            if (rwall != lwall) {
                part.assign(lwall, rwall);

                char* tmp = NULL;

                // Try to convert the string
                Idna_rc idnaRet = static_cast<Idna_rc>( idna_to_ascii_8z (part.c_str (), &tmp, 0) );
                if (idnaRet != IDNA_SUCCESS)
                {
                    if ( tmp ) free(tmp);
                    throw std::runtime_error (std::string ("IDNA error: '") + idna_strerror (idnaRet) + "', str=" + std::string(strBeg, strEnd) );
                }

                converted.append (tmp);
                free(tmp);
            }

            firstRun = false;
        }

        // Further "normalize" the string by lowering its case
        to_lower (converted);

        return true;
    }


    // returns true if result is in converted, false - str does not need conversion
    static bool IdnaDecode (std::string::const_iterator strBeg, std::string::const_iterator strEnd, std::string& converted)
    {
        const std::string prefix("xn--");

        // check if need conversion
        if ( std::search(strBeg, strEnd, prefix.begin(), prefix.end()) == strEnd )
            return false;

        // Split into DNS parts and conver them independently
        // because each part is limited in length: it may not exceed 63 characters
        converted.clear();
        converted.reserve ( (strEnd-strBeg) * 3);

        bool firstRun = true;
        std::string part;
        std::string::const_iterator rwall = strBeg;
        for (;;) {

            // End of string
            if (rwall == strEnd)
                break;

            // Move onto the next part, which must begin with a dot in all
            // cases except for the very first one
            if (!firstRun)
                converted.push_back ('.');

            // Next part turns out to be just the dot  and nothing more
            if (++rwall == strEnd)
                break;

            // Find right wall for the current part
            std::string::const_iterator lwall = firstRun ? strBeg : rwall;
            rwall = std::find (lwall, strEnd, '.');

            // If it's empty simply skip over
            if (rwall != lwall) {
                part.assign(lwall, rwall);

                char* tmp = NULL;

                // Try to convert the string
                Idna_rc idnaRet = static_cast<Idna_rc>( idna_to_unicode_8z8z (part.c_str (), &tmp, 0) );
                if (idnaRet != IDNA_SUCCESS)
                {
                    if ( tmp ) free(tmp);
                    throw std::runtime_error (std::string ("IDNA error: '") + idna_strerror (idnaRet) + "', str=" + std::string(strBeg, strEnd) );
                }

                converted.append (tmp);
                free(tmp);
            }

            firstRun = false;
        }

        return true;
    }

    // encode entire string to punycode
    const std::string& encode(const std::string& str, std::string& converted)
    {
        if (IdnaEncode(str.begin(), str.end(), converted)) return converted;
        else return str;    // no change
    }

    // decode entire string as punycode
    const std::string& decode(const std::string& str, std::string& converted)
    {
        if (IdnaDecode(str.begin(), str.end(), converted)) return converted;
        else return str;
    }

    // encode address to punycode
    const std::string& encodeAddr(const std::string& addr, std::string& converted)
    {
        std::string::const_iterator it = std::find(addr.begin(), addr.end(), '@');
        if ( it == addr.end() || ++it == addr.end())
            return addr;    // no @ or ends with @

        std::string tmp;
        if (IdnaEncode(it, addr.end(), tmp)) {
            converted.clear();
            converted.reserve( tmp.size() + (it-addr.begin()) + 1 );
            converted.assign(addr.begin(), it);
            converted.append(tmp);
            return converted;
        } else return addr; // no conversion needed
    }

    // decode address as punycode
    const std::string& decodeAddr(const std::string& addr, std::string& converted)
    {
        std::string::const_iterator it = std::find(addr.begin(), addr.end(), '@');
        if ( it == addr.end() || ++it == addr.end())
            return addr;    // no @ or ends with @

        std::string tmp;
        if (IdnaDecode(it, addr.end(), tmp)) {
            converted.clear();
            converted.reserve( tmp.size() + (it-addr.begin()) + 1 );
            converted.assign(addr.begin(), it);
            converted.append(tmp);
            return converted;
        } else return addr; // no conversion needed
    }

} // namespace idn

// vi: expandtab:sw=4:ts=4
// kate: replace-tabs on; indent-width 4; tab-width 4;
