#include <sstream>
#include "xconfig.h"
#include "initexc.h"

namespace bb {
namespace xmlConfig {

XConfig::~XConfig()
{
    if(0 != doc) {
        xmlFreeDoc(doc);
    }
}

int
XConfig::Parse()
{
    const char * config_file = getenv("YANDEX_XML_CONFIG");
    return realparse(config_file ? config_file : "./servant.cfg");
}

int
XConfig::realparse(const char * filename)
{
#ifndef __WIN32__
    if(0 != access(filename, R_OK))
    {
        int error = errno;
        return error;
    }
#endif
    xmlParserCtxtPtr pctx = xmlNewParserCtxt();
    if(0 == pctx) {
        return ENOMEM;
    }
    doc = xmlCtxtReadFile(pctx, filename, 0, XML_PARSE_NOCDATA);

    if(0 == doc) {
        return EDOM;
    }
    int ret = xmlXIncludeProcessFlags(doc,
            XML_PARSE_XINCLUDE | XML_PARSE_NOCDATA | XML_PARSE_NOXINCNODE);
    if(0 > ret) {
        return ret;
    }
    xmlFreeParserCtxt(pctx);
    return 0;
}

int
XConfig::realparse(const std::string &xmltext)
{
    doc = xmlReadMemory(xmltext.data(), static_cast<int>(xmltext.size()), NULL, NULL, XML_PARSE_NOCDATA);
    if(0 == doc) {
        return EDOM;
    }
    return 0;
}

Parts
XConfig::GetParts(const char * xpathExpr)
{
    return Parts(doc, xpathExpr);
}

Part
XConfig::GetFirst(const char * xpathExpr)
{
    Parts parts(doc, xpathExpr);
    if(0 == parts.Size())
    {
        std::stringstream ss_;
        ss_ << "in GetFirst no parts for </+" << xpathExpr << ">";
        throw(initexc(ss_.str()));
    }
    return Part(doc, parts.xpathObj->nodesetval->nodeTab[0]);
}

bool
XConfig::GetIfExists(const char * xpathExpr, long int & variable) {
    Parts parts(doc, xpathExpr);
    if(0 == parts.Size()) {
        return false;
    }
    xmlChar * ctp = xmlNodeGetContent(parts.xpathObj->nodesetval->nodeTab[0]);
    if(0 == ctp) return false;
    variable = strtol(BAD_REV_CAST(ctp),0,0);
    xmlFree(ctp);
    return true;
}

bool
XConfig::GetIfExists(const char * xpathExpr, std::string & variable) {
    Parts parts(doc, xpathExpr);
    if(0 == parts.Size()) {
        return false;
    }
    xmlChar * ctp = xmlNodeGetContent(parts.xpathObj->nodesetval->nodeTab[0]);
    if(0 == ctp) {
        variable.resize(0);
    }
    else {
        variable.assign(BAD_REV_CAST(ctp));
    }
    xmlFree(ctp);
    return true;
}

Parts::Parts(xmlDocPtr doc_, const char * xpathExpr):
    doc(doc_),
    xpathCtx(xmlXPathNewContext(doc)),
    xpathObj(0),
    size(0)
{
    if(0 == xpathCtx)
    {
        //E("Can't initialize context");
    }
    evaluate(xpathExpr);
}

Parts::Parts(xmlDocPtr doc_, xmlNodePtr startnode, const char * xpathExpr):
    doc(doc_),
    xpathCtx(xmlXPathNewContext(doc)),
    xpathObj(0),
    size(0)
{
    if(0 == xpathCtx)
    {
        //E("Can't initialize context");
    }
    setcurnode(startnode);
    evaluate(xpathExpr);
}

int
Parts::setcurnode(xmlNodePtr node)
{
    xpathCtx->node = node;
    return 0;
}

/*
 * Copy ctor optimized to return Parts instances by value from functions.
 * It modifies right hand side expression so dtor doesn't delete xml tree
 * parts we are still working with. Thus Parts instances steal ownership of
 * underlying xml tree objects when copy constructed.
 * If we want to perform honest copying we must copy xml tree parts as well,
 * which is unnecessary in our case.
 */
Parts::Parts(const Parts & rhs):
    doc(rhs.doc),
    xpathCtx(rhs.xpathCtx),
    xpathObj(rhs.xpathObj),
    size(rhs.size)
{
    //Parts & parts = const_cast<Parts &>(rhs);
    const Parts & parts = rhs;
    parts.xpathCtx = 0;
    parts.xpathObj = 0;
}

Parts::~Parts()
{
    if(0 != xpathObj)
        xmlXPathFreeObject(xpathObj);
    if(0 != xpathCtx)
        xmlXPathFreeContext(xpathCtx);
}

int
Parts::evaluate(const char * xpathExpr)
{
    xpathObj = xmlXPathEvalExpression(reinterpret_cast<xmlChar*>(const_cast<char*>(xpathExpr)), xpathCtx);
    if(0 == xpathObj)
    {
        //E("Can't eval xpath expression" << xpathExpr);
        return EDOM;
    }
    xmlNodeSetPtr nodes = xpathObj->nodesetval;
    size = (nodes) ? nodes->nodeNr : 0;
    return 0;
}

Part
Parts::operator[](int number)
{
    return Part(doc, xpathObj->nodesetval->nodeTab[number]);
}

Part::Part(const Part & rhs):
    doc(rhs.doc),
    thisnode(rhs.thisnode)
{
}

Part::Part(xmlDocPtr doc_, xmlNodePtr node):
    doc(doc_),
    thisnode(node)
{
}

Parts
Part::GetParts(const char * xpathExpr)
{
    return Parts(doc, thisnode, xpathExpr);
}

Part
Part::GetFirst(const char * xpathExpr)
{
    Parts parts(doc, thisnode, xpathExpr);
    if(0 == parts.Size())
    {
        std::stringstream ss_;
        ss_ << "in GetFirst no parts for <";
        xmlChar * path = xmlGetNodePath(thisnode);
        if(0 != path) {
            ss_ << "+" << BAD_REV_CAST(path);
            xmlFree(path);
        }
        ss_ << xpathExpr << ">";
        throw(initexc(ss_.str()));
    }
    return Part(doc, parts.xpathObj->nodesetval->nodeTab[0]);
}

bool
Part::GetIfExists(const char * xpathExpr, long int & variable) {
    Parts parts(doc, thisnode, xpathExpr);
    if(0 == parts.Size()) {
        return false;
    }
    xmlChar * ctp = xmlNodeGetContent(parts.xpathObj->nodesetval->nodeTab[0]);
    if(0 == ctp) return false;
    variable = strtol(BAD_REV_CAST(ctp),0,0);
    xmlFree(ctp);
    return true;
}

bool
Part::GetIfExists(const char * xpathExpr, std::string & variable) {
    Parts parts(doc, thisnode, xpathExpr);
    if(0 == parts.Size()) {
        return false;
    }
    xmlChar * ctp = xmlNodeGetContent(parts.xpathObj->nodesetval->nodeTab[0]);
    if(0 == ctp) {
        variable.resize(0);
    }
    else {
        variable.assign(BAD_REV_CAST(ctp));
    }
    xmlFree(ctp);
    return true;
}

std::string
Part::GetName()
{
    std::string aux(BAD_REV_CAST(thisnode->name));
    return aux;
}


std::string
Part::asString()
{
    xmlChar * ctp = xmlNodeGetContent(thisnode);
    if(0 == ctp) return std::string();
    std::string aux(BAD_REV_CAST(ctp));
    xmlFree(ctp);
    return aux;
}

long int
Part::asLong(long int defval)
{
    xmlChar * ctp = xmlNodeGetContent(thisnode);
    if(0 == ctp) return defval;
    long int aux = strtol(BAD_REV_CAST(ctp),0,0);
    xmlFree(ctp);
    return aux;
}

}; // namespace xmlConfig
}; // namespace bb
