#include <iterator>
#include <iostream>
#include <fstream>
#include <algorithm>
#include <boost/function.hpp>
#include <boost/functional.hpp>
#include <boost/iterator/transform_iterator.hpp>
#include <boost/unordered_map.hpp>
#include <boost/bind.hpp>
#include <boost/fusion/include/adapt_struct.hpp>
#include <boost/spirit/include/qi.hpp>
#include <boost/spirit/include/phoenix.hpp>
#include <boost/spirit/include/support_multi_pass.hpp>
#include <boost/spirit/include/phoenix_stl.hpp>
#include <boost/spirit/include/phoenix_fusion.hpp>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/range/algorithm/for_each.hpp>
#include <boost/range/algorithm/find_if.hpp>

#include <sys/types.h>
#include <ifaddrs.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <hostsearch/network.hpp>

BOOST_FUSION_ADAPT_STRUCT(
  hostsearch::named_network,
  (boost::asio::ip::address, second.ip)
  (int, second.prefix_bits)
  (std::string, first)
)

namespace hostsearch{
namespace detail {

template <int address_size>
bool addr_belongs_to_network(const uint8_t* lh, const uint8_t* rh, int n_bits)
{
  assert(n_bits >= 0 && n_bits <= address_size * 8);

  for (int z = n_bits / 8; z > 0; --z)
    if (*lh++ != *rh++)
      return false;

  if (int m = n_bits%8)
  {
    uint8_t mask = ~uint8_t((1 << (8-m)) -1);
    if (((*lh++) & mask) != ((*rh++) & mask))
      return false;
  }

  return true;
}

// Efficient specialization for ipv4 case
inline bool addr_belongs_to_network(unsigned long lh, unsigned long rh, int n_bits)
{
  assert(n_bits >= 0 && n_bits <= 32);
  return ((lh ^ rh) >> (32 - n_bits)) == 0;
}


inline bool addr_belongs_to_network(const boost::asio::ip::address& ip,
    const network& network)
{
  if (ip.is_v4() && network.ip.is_v4())
    return addr_belongs_to_network(ip.to_v4().to_ulong(),
        network.ip.to_v4().to_ulong(), network.prefix_bits);

  if (ip.is_v6() && network.ip.is_v6())
    return addr_belongs_to_network<16>(ip.to_v6().to_bytes().data(),
        network.ip.to_v6().to_bytes().data(), network.prefix_bits);

  return false;
}

inline bool addr_belongs_to_named_network(const boost::asio::ip::address& ip,
    const named_network& network)
{
  return addr_belongs_to_network(ip, network.second);
}

template <class AddressIt>
bool addr_belong_to_named_network(AddressIt beg, AddressIt end,
    const named_network& network)
{
  return end != std::find_if(beg, end,
      boost::bind(&addr_belongs_to_named_network, _1, network));
}

template <class NamedNetworkIterator>
inline bool addr_belongs_to_named_networks(const boost::asio::ip::address& ip,
    NamedNetworkIterator beg, NamedNetworkIterator end)
{
  return end !=
    std::find_if(beg, end, boost::bind(&addr_belongs_to_named_network, ip, _1));
}

template <class AddressIt, class NamedNetworkIterator>
bool addr_belong_to_named_networks(AddressIt addr_beg, AddressIt addr_end,
    NamedNetworkIterator beg, NamedNetworkIterator end)
{
  return addr_end !=
    std::find_if(addr_beg, addr_end,
        boost::bind(
          &addr_belongs_to_named_networks<NamedNetworkIterator>, _1, beg, end));
}

typedef boost::asio::ip::tcp::resolver resolver_t;
resolver_t::query
inline parse_hostname(const std::string& url)
{
  std::size_t p1 = url.find("://");
  p1 = (p1 == std::string::npos ? 0 : p1+3);
  std::size_t p2 = url.find("/", p1);
  p2 = (p2 == std::string::npos ? url.length() : p2);
  std::size_t p3 = url.find(":", p1);
  p2 = (p3 == std::string::npos ? p2 : (p3 < p2 ? p3 : p2));

  std::string host( url.begin() + p1, url.begin() + std::min(p2, p3) );
  return resolver_t::query(host, "");
}

inline bool host_belongs_to_named_network(const std::string& url,
    const named_network& network)
{
  namespace ip = boost::asio::ip;
  typedef ip::address address_t;
  typedef ip::tcp::endpoint endpoint_t;

  boost::asio::io_service ios;
  resolver_t r(ios);

  address_t addr = r.resolve( parse_hostname( url ) )->endpoint().address();

  return addr_belongs_to_named_network( addr, network );
}

template <class NamedNetworkIterator>
inline bool host_belongs_to_named_networks(const std::string& url,
    NamedNetworkIterator beg, NamedNetworkIterator end)
{
  namespace ip = boost::asio::ip;
  typedef ip::address address_t;
  typedef ip::tcp::endpoint endpoint_t;

  boost::asio::io_service ios;
  resolver_t r(ios);

  // Resolve hostname to a list of ip addresses
  resolver_t::iterator addr_beg = r.resolve( parse_hostname(url) );
  resolver_t::iterator addr_end;

  boost::function<address_t(const endpoint_t&)> take_address =
    boost::mem_fn<address_t>(&endpoint_t::address);

  return addr_belong_to_named_networks(
    boost::make_transform_iterator(addr_beg, take_address),
    boost::make_transform_iterator(addr_end, take_address),
    beg, end);
}

template <class NamedNetworksRange>
inline bool host_belongs_to_named_networks(const std::string& url,
    NamedNetworksRange nets)
{
    return host_belongs_to_named_networks( url, boost::begin( nets ), boost::end( nets ) );
}

class address_iterator
  : public boost::iterator_facade<
      address_iterator,
      boost::asio::ip::address,
      boost::forward_traversal_tag,
      boost::asio::ip::address>
{
public:
  address_iterator() : a_(0) {}

  static address_iterator create(ifaddrs* a)
  {
    address_iterator it;
    it.a_ = a;
    return it;
  }

private:
  friend class boost::iterator_core_access;

  void increment()
  {
    while ((a_ = a_->ifa_next))
    {
      if (a_->ifa_addr->sa_family == AF_INET
          || a_->ifa_addr->sa_family == AF_INET6)
        return;
    }
    *this = address_iterator();
  }

  bool equal(const address_iterator& other) const
  { return a_ == other.a_; }

  boost::asio::ip::address dereference() const
  {
    if (a_->ifa_addr->sa_family == AF_INET)
    {
      return boost::asio::ip::address_v4(
        ntohl(
          reinterpret_cast<struct sockaddr_in*>(a_->ifa_addr)->sin_addr.s_addr));
    }
    else
    {
      uint8_t* addr =
        reinterpret_cast<struct sockaddr_in6*>(a_->ifa_addr)->sin6_addr.s6_addr;
      boost::asio::ip::address_v6::bytes_type raw_bytes;
      std::copy(addr, addr+raw_bytes.size(), raw_bytes.data());
      return boost::asio::ip::address_v6(raw_bytes);
    }
  }

  ifaddrs* a_;
};

inline address_iterator find_local_addresses()
{
  ifaddrs* addrs = 0;
  getifaddrs(&addrs);
  return address_iterator::create(addrs);
}

typedef boost::function<bool(const named_network&)> NetworkPredicate;

template<typename NamedNetworksRange>
struct nets_finder {
    typedef boost::filtered_range< NetworkPredicate, const NamedNetworksRange > result_range;

    nets_finder(const NetworkPredicate& p) : pred( p )
    {}
    result_range operator()(const NamedNetworksRange& nets) const {
      return boost::adaptors::filter( nets, pred );
    }
    NetworkPredicate pred;
};

inline NetworkPredicate getLocalNetsPredicate() {
  address_iterator addr_beg = find_local_addresses();
  address_iterator addr_end;
  return boost::bind(
      &addr_belong_to_named_network<address_iterator>,
      addr_beg, addr_end, _1 );
}

template<typename NamedNetworksRange>
bool networkHasCommonName( const NamedNetworksRange& filterRange, const named_network& net ) {
    return boost::find_if( filterRange, boost::bind( &named_network::first, _1) == net.first )
        != boost::end( filterRange );
}

template<typename NamedNetworksRange>
NetworkPredicate getSameNamedNetsPredicate( const NamedNetworksRange& filterRange ) {
    return boost::bind( &networkHasCommonName<NamedNetworksRange>, filterRange, _1 );
}

struct ip_parser_impl
{
  template <typename T>
  struct result { typedef boost::asio::ip::address type; };

  boost::asio::ip::address operator()(const std::string& ip) const
  {
    try
    {
      return boost::asio::ip::address::from_string(ip);
    }
    catch (...)
    {
      std::ostringstream oss;
      oss << "failed to parse ip address: " << ip;
      throw std::runtime_error(oss.str().c_str());
    }
  }
};

template <typename Iterator>
struct named_network_grammar :
  boost::spirit::qi::grammar<Iterator, boost::spirit::qi::space_type,
    named_network()>
{
  named_network_grammar() : named_network_grammar::base_type(nn)
  {
    namespace qi = boost::spirit::qi;
    using boost::phoenix::at_c;

    nn =
      ip[at_c<0>(qi::_val) = ip_parser(qi::_1)]
      >> qi::ushort_[at_c<1>(qi::_val) = qi::_1]
      >> name[at_c<2>(qi::_val) = qi::_1];

    name = qi::raw[+qi::alnum];
    ip = qi::raw[+qi::char_("0-9a-f:.")];
  }

  boost::spirit::qi::rule<Iterator,
      boost::spirit::qi::space_type, named_network()> nn;
  boost::spirit::qi::rule<Iterator, std::string()> name;
  boost::spirit::qi::rule<Iterator, std::string()> ip;
  boost::phoenix::function<ip_parser_impl> ip_parser;
};


inline std::vector<hostsearch::named_network>
parse_networks(std::istream& networks)
{
  std::vector<hostsearch::named_network> vn;

  namespace spirit = boost::spirit;
  namespace qi = spirit::qi;

  typedef std::istreambuf_iterator<char> base_iterator_t;
  typedef spirit::multi_pass<base_iterator_t> iterator_t;

  iterator_t beg = spirit::make_default_multi_pass(base_iterator_t(networks));
  iterator_t end = spirit::make_default_multi_pass(base_iterator_t());
  detail::named_network_grammar<iterator_t> nn_gr;

  if (!spirit::qi::phrase_parse(beg, end, +nn_gr, qi::space, vn)
      || beg != end)
  {
    std::ostringstream oss;
    oss << "failed to parse networks, context: " <<
      boost::make_iterator_range(beg, end);
    throw std::runtime_error(oss.str().c_str());
  }

  return vn;
}

inline std::vector<std::string>
parse_hosts(std::istream& hosts)
{
  std::vector<std::string> vh;
  std::copy(
    std::istream_iterator<std::string>(hosts),
    std::istream_iterator<std::string>(),
    std::back_inserter(vh));
  return vh;
}

// Tests if i is a subnet of j
inline bool is_subnet(const named_network& i, const named_network& j)
{
  return i.second.prefix_bits >= j.second.prefix_bits
    && addr_belongs_to_named_network(i.second.ip, j);
}

} // namespace detail {

template< typename NamedNetworksRange >
typename detail::nets_finder< NamedNetworksRange >::result_range
findLocalNets( const NamedNetworksRange& nets ) {
    return detail::nets_finder< NamedNetworksRange >(
            detail::getLocalNetsPredicate() )( nets );
}

template< typename NamedNetworksRange >
typename detail::nets_finder< NamedNetworksRange >::result_range
findNonlocalNets( const NamedNetworksRange& nets ) {
    return detail::nets_finder< NamedNetworksRange >(
            boost::not1( detail::getLocalNetsPredicate() ) )( nets );
}

template< typename NamedNetworksRange, typename FilterNetworksRange >
typename detail::nets_finder< NamedNetworksRange >::result_range
findSameNamedNets( const NamedNetworksRange& nets, const FilterNetworksRange& filter ) {
    return detail::nets_finder< NamedNetworksRange >(
            detail::getSameNamedNetsPredicate( filter ) )( nets );
}

template <class NamedNetworksRange, class HostsRange>
typename HostsRange::iterator find_core( NamedNetworksRange nets,
    HostsRange hosts )
{
  typedef detail::nets_finder<NamedNetworksRange> nets_finder;
  typedef typename nets_finder::result_range nets_range;

  nets_range local_nets = findLocalNets( nets );

  typename HostsRange::iterator local_host = std::find_if( boost::begin( hosts ),
      boost::end( hosts ), boost::bind(&detail::host_belongs_to_named_networks<nets_range>,
          _1, local_nets )
  );
  if( local_host != boost::end( hosts ) ) {
      return local_host;
  }

  return std::find_if( boost::begin( hosts ),
      boost::end( hosts ), boost::bind(&detail::host_belongs_to_named_networks<nets_range>,
          _1, findSameNamedNets( nets, local_nets ) )
  );
}

template <class NamedNetworkIterator, class HostIterator>
HostIterator find(NamedNetworkIterator net_begin,
    NamedNetworkIterator net_end,
    HostIterator host_begin,
    HostIterator host_end)
{
  return find_core(
      boost::make_iterator_range( net_begin, net_end ),
      boost::make_iterator_range( host_begin, host_end )
  );
}

inline std::string find(std::istream& networks, std::istream& hosts)
{
  std::vector<hostsearch::named_network> vn;
  std::vector<std::string> vh;

  try {
    detail::parse_networks(networks).swap(vn);
    detail::parse_hosts(hosts).swap(vh);

  } catch (std::exception& e) {
    std::ostringstream oss;
    oss << "parse error: " << e.what();
    throw std::runtime_error(oss.str().c_str());
  }

  std::vector<std::string>::iterator hostit =
    hostsearch::find(vn.begin(), vn.end(), vh.begin(), vh.end());

  return hostit == vh.end() ? std::string() : *hostit;
}

inline std::string find(const char* networks_filename,
    const char* hosts_filename)
{
  std::ifstream nets(networks_filename);
  if (!nets.is_open())
    throw std::runtime_error(
      std::string("failed to open dc config ").append(networks_filename));

  std::ifstream hosts(hosts_filename);
  if (!hosts.is_open())
    throw std::runtime_error(
      std::string("failed to open hosts ").append(hosts_filename));

  return hostsearch::find(nets, hosts);
}

template <class NamedNetworksRange, class HostsRange, typename Handler>
void associate_core(const NamedNetworksRange& networks,
        const HostsRange& hosts,
        Handler handle)
{

  struct ApplyHandleForAppropriateNetworks
  {
    typedef typename detail::nets_finder<NamedNetworksRange>::result_range nets_range;

    ApplyHandleForAppropriateNetworks( Handler h, const NamedNetworksRange& nets )
        : h_(h), nets_( findNonlocalNets( nets ) )
    {

    }
    void operator() ( const std::string& host )
    {
      using namespace boost;
      for_each( nets_ | adaptors::filtered( bind( detail::host_belongs_to_named_network, host, _1 ) ),
          bind<void>( h_, _1, host )
      );
    }

    Handler h_;
    nets_range nets_;
  } handle_hook( handle, networks );

  boost::for_each( hosts, handle_hook );
}

template <typename Handler>
void associate(std::istream& networks, std::istream& hosts, Handler handle)
{
  std::vector<hostsearch::named_network> vn;
  std::vector<std::string> vh;

  try {
    detail::parse_networks(networks).swap(vn);
    detail::parse_hosts(hosts).swap(vh);

  } catch (std::exception& e) {
    std::ostringstream oss;
    oss << "parse error: " << e.what();
    throw std::runtime_error(oss.str().c_str());
  }

  hostsearch::associate_core(vn, vh, handle);
}

template <typename Handler>
void associate(const char* networks_filename,
        const char* hosts_filename,
        Handler handle)
{
  std::ifstream nets(networks_filename);
  if (!nets.is_open())
    throw std::runtime_error(
      std::string("failed to open dc config ").append(networks_filename));

  std::ifstream hosts(hosts_filename);
  if (!hosts.is_open())
    throw std::runtime_error(
      std::string("failed to open hosts ").append(hosts_filename));

  hostsearch::associate(nets, hosts, handle);
}


template <class NamedNetworkIterator>
std::vector<NamedNetworkIterator> canonize_namednetworks(
  NamedNetworkIterator beg, NamedNetworkIterator end)
{
  // Eliminate duplicates, eliminate subnets.
  std::set<NamedNetworkIterator> nns;
  std::vector<NamedNetworkIterator> r;
  for (NamedNetworkIterator i=beg; i!=end; ++i)
  {
    for (NamedNetworkIterator j=i+1; j!=end; ++j)
    {
      if (detail::is_subnet(*i, *j))
        nns.insert(i);
      else if (detail::is_subnet(*j, *i))
        nns.insert(j);
    }
    if (nns.find(i) == nns.end())
      r.push_back(i);
  }

  return r;
}

inline std::string canonize_namednetworks(std::istream& networks)
{
  std::vector<hostsearch::named_network> vn;

  try {
    detail::parse_networks(networks).swap(vn);
  } catch (std::exception& e) {
    std::ostringstream oss;
    oss << "parse error: " << e.what();
    throw std::runtime_error(oss.str().c_str());
  }

  typedef std::vector<hostsearch::named_network>::iterator iterator_t;
  std::vector<iterator_t> vn_c  = canonize_namednetworks(vn.begin(), vn.end());

  std::ostringstream oss;
  for (iterator_t nn: vn_c)
    oss << nn->second.ip << '\t' << nn->second.prefix_bits << '\t' << nn->first
        << std::endl;

  return oss.str();
}

inline std::string canonize_namednetworks(const char* networks_filename)
{
  std::ifstream nets(networks_filename);
  if (!nets.is_open())
    throw std::runtime_error(
      std::string("failed to open dc config ").append(networks_filename));

  return canonize_namednetworks(nets);
}

} // namespace hostsearch {
