package ru.yandex.cache.async;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.EmptyFutureCallback;

public class AsyncCache<K, V, C, E extends Exception>
    implements GenericAutoCloseable<E>
{
    private final Map<K, QueuedRequests<V>> queuedRequests =
        new ConcurrentHashMap<>();
    private final AsyncStorage<K, V, E> storage;
    private final AsyncCacheTtlCalculator<? super K, ? super V> ttlCalculator;
    private final AsyncLoader<K, V, C> loader;

    public AsyncCache(
        final AsyncStorage<K, V, E> storage,
        final AsyncCacheTtlCalculator<? super K, ? super V> ttlCalculator,
        final AsyncLoader<K, V, C> loader)
    {
        this.storage = storage;
        this.ttlCalculator = ttlCalculator;
        this.loader = loader;
    }

    @Override
    public void close() throws E {
        storage.close();
    }

    public void get(
        final K key,
        final C context,
        final FutureCallback<? super AsyncCacheResult<V>> callback)
    {
        storage.get(
            key,
            new StorageCallback<>(callback, this, key, context));
    }

    private static class QueuedRequests<V>
        extends ArrayList<FutureCallback<? super AsyncCacheResult<V>>>
    {
        private static final long serialVersionUID = 0L;

        QueuedRequests() {
            super(4);
        }

        QueuedRequests(final QueuedRequests<V> queuedRequests) {
            super(queuedRequests);
        }
    }

    private static class StorageCallback<K, V, C>
        extends AbstractFilterFutureCallback<V, AsyncCacheResult<V>>
    {
        private final AsyncCache<K, V, C, ?> cache;
        private final K key;
        private final C context;

        StorageCallback(
            final FutureCallback<? super AsyncCacheResult<V>> callback,
            final AsyncCache<K, V, C, ?> cache,
            final K key,
            final C context)
        {
            super(callback);
            this.cache = cache;
            this.key = key;
            this.context = context;
        }

        @Override
        public void completed(final V value) {
            if (value == null) {
                QueuedRequests<V> newQueuedRequests = new QueuedRequests<>();
                newQueuedRequests.add(callback);
                QueuedRequests<V> oldQueuedRequests =
                    cache.queuedRequests.putIfAbsent(key, newQueuedRequests);
                if (oldQueuedRequests == null) {
                    cache.loader.load(
                        key,
                        context,
                        new LoaderCallback<>(cache, key));
                } else {
                    synchronized (oldQueuedRequests) {
                        if (!oldQueuedRequests.isEmpty()) {
                            oldQueuedRequests.add(callback);
                            return;
                        }
                    }
                    // Old queued requests already empty
                    // Request was just finished, try again
                    cache.storage.get(key, this);
                }
            } else {
                callback.completed(
                    new AsyncCacheResult<>(value, AsyncCacheResultType.HIT));
            }
        }
    }

    private static class LoaderCallback<K, V> implements FutureCallback<V> {
        private final AsyncCache<K, V, ?, ?> cache;
        private final K key;

        LoaderCallback(final AsyncCache<K, V, ?, ?> cache, final K key) {
            this.cache = cache;
            this.key = key;
        }

        private List<FutureCallback<? super AsyncCacheResult<V>>>
            queuedRequestsCopy()
        {
            QueuedRequests<V> queuedRequests =
                cache.queuedRequests.remove(key);
            if (queuedRequests == null) {
                return Collections.emptyList();
            }
            QueuedRequests<V> copy;
            synchronized (queuedRequests) {
                copy = new QueuedRequests<>(queuedRequests);
                queuedRequests.clear();
            }
            return copy;
        }

        @Override
        public void cancelled() {
            List<FutureCallback<? super AsyncCacheResult<V>>> queuedRequests =
                queuedRequestsCopy();
            int size = queuedRequests.size();
            for (int i = 0; i < size; ++i) {
                queuedRequests.get(i).cancelled();
            }
        }

        @Override
        public void failed(final Exception e) {
            List<FutureCallback<? super AsyncCacheResult<V>>> queuedRequests =
                queuedRequestsCopy();
            int size = queuedRequests.size();
            for (int i = 0; i < size; ++i) {
                queuedRequests.get(i).failed(e);
            }
        }

        @Override
        public void completed(final V value) {
            long ttl = cache.ttlCalculator.ttlFor(key, value);
            if (ttl > 0L) {
                cache.storage.put(
                    key,
                    value,
                    ttl,
                    EmptyFutureCallback.instance());
            }
            List<FutureCallback<? super AsyncCacheResult<V>>> queuedRequests =
                queuedRequestsCopy();
            int size = queuedRequests.size();
            queuedRequests.get(0).completed(
                new AsyncCacheResult<>(value, AsyncCacheResultType.MISS));
            if (size > 1) {
                AsyncCacheResult<V> result =
                    new AsyncCacheResult<>(value, AsyncCacheResultType.LOCK);
                for (int i = 1; i < size; ++i) {
                    queuedRequests.get(i).completed(result);
                }
            }
        }
    }
}

