package ru.yandex.cache.async;

import com.googlecode.concurrentlinkedhashmap.ConcurrentLinkedHashMap;
import com.googlecode.concurrentlinkedhashmap.EntryWeigher;
import com.googlecode.concurrentlinkedhashmap.Weigher;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.util.timesource.TimeSource;

public class ConcurrentLinkedHashMapAsyncStorage<K, V>
    implements AsyncStorage<K, V, RuntimeException>
{
    public static final int MEMBER_WEIGHT = 8;
    public static final int OBJECT_WEIGHT = 48;

    private final ConcurrentLinkedHashMap<K, Entry<V>> storage;

    public ConcurrentLinkedHashMapAsyncStorage(
        final long capacity,
        final int concurrency,
        final Weigher<? super K> keyWeigher,
        final Weigher<? super V> valueWeigher)
    {
        storage = new ConcurrentLinkedHashMap.Builder<K, Entry<V>>()
            .maximumWeightedCapacity(capacity)
            .concurrencyLevel(concurrency)
            .weigher(new StorageWeigher<>(keyWeigher, valueWeigher))
            .build();
    }

    public int size() {
        return storage.size();
    }

    public long weightedSize() {
        return storage.weightedSize();
    }

    @Override
    public void close() {
        storage.setCapacity(0);
    }

    @Override
    public void get(final K key, final FutureCallback<? super V> callback) {
        Entry<V> entry = storage.get(key);
        V value;
        if (entry == null) {
            value = null;
        } else {
            long now = TimeSource.INSTANCE.currentTimeMillis();
            if (entry.expireTimestamp < now) {
                storage.remove(key, entry);
                value = null;
            } else {
                value = entry.value;
            }
        }
        callback.completed(value);
    }

    @Override
    public void put(
        final K key,
        final V value,
        final long ttl,
        final FutureCallback<Void> callback)
    {
        long now = TimeSource.INSTANCE.currentTimeMillis();
        storage.put(key, new Entry<>(value, now + ttl));
        callback.completed(null);
    }

    public void remove(final K key) {
        storage.remove(key);
    }

    private static class Entry<V> {
        private final V value;
        private final long expireTimestamp;

        Entry(final V value, final long expireTimestamp) {
            this.value = value;
            this.expireTimestamp = expireTimestamp;
        }
    }

    private static class StorageWeigher<K, V>
        implements EntryWeigher<K, Entry<V>>
    {
        private final Weigher<? super K> keyWeigher;
        private final Weigher<? super V> valueWeigher;

        StorageWeigher(
            final Weigher<? super K> keyWeigher,
            final Weigher<? super V> valueWeigher)
        {
            this.keyWeigher = keyWeigher;
            this.valueWeigher = valueWeigher;
        }

        @Override
        public int weightOf(final K key, final Entry<V> entry) {
            return keyWeigher.weightOf(key)
                + valueWeigher.weightOf(entry.value)
                + MEMBER_WEIGHT // expireTimestamp
                + OBJECT_WEIGHT // Entry
                + OBJECT_WEIGHT // storage entry
                ;
        }
    }
}

