package ru.yandex.blackbox;

import java.io.IOException;

import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

public class BlackboxAddress
    extends AbstractStringBuilderable
    implements JsonValue
{
    private static final DateTimeFormatter FORMATTER =
        DateTimeFormat.forPattern("yyyy-MM-dd' 'HH:mm:ss")
            .withZone(DateTimeZone.forID("Europe/Moscow"));
    private static final String ADDRESS = "address";
    private static final String BORN_DATE = "born-date";
    private static final String DEFAULT = "default";
    private static final String NATIVE = "native";
    private static final String RPOP = "rpop";
    private static final String UNSAFE = "unsafe";
    private static final String VALIDATED = "validated";

    private final String email;
    private final long bornDate;
    private final boolean defaultFlag;
    private final boolean nativeFlag;
    private final boolean rpopFlag;
    private final boolean unsafeFlag;
    private final boolean validatedFlag;

    public BlackboxAddress(final JsonMap map)
        throws BlackboxMalformedResponseException
    {
        try {
            email = map.getString(ADDRESS);
            bornDate = map.get(BORN_DATE, x -> parseBornDate(x));
            defaultFlag = map.getBoolean(DEFAULT);
            nativeFlag = map.getBoolean(NATIVE);
            rpopFlag = map.getBoolean(RPOP);
            unsafeFlag = map.getBoolean(UNSAFE);
            validatedFlag = map.getBoolean(VALIDATED);
        } catch (JsonException e) {
            throw new BlackboxMalformedResponseException(
                "Failed to parse address info from: " + map,
                e);
        }
    }

    private static long parseBornDate(final String bornDate) {
        try {
            return FORMATTER.parseDateTime(bornDate).getMillis();
        } catch (IllegalArgumentException e) {
            return 0L;
        }
    }

    public String email() {
        return email;
    }

    public long bornDate() {
        return bornDate;
    }

    public boolean defaultFlag() {
        return defaultFlag;
    }

    public boolean nativeFlag() {
        return nativeFlag;
    }

    public boolean rpopFlag() {
        return rpopFlag;
    }

    public boolean unsafeFlag() {
        return unsafeFlag;
    }

    public boolean validatedFlag() {
        return validatedFlag;
    }

    public static void appendIf(
        final StringBuilder sb,
        final boolean flag,
        final String value)
    {
        if (flag) {
            sb.append(',');
            sb.append(value);
        }
    }

    public static void appendIf(
            final StringBuilder sb,
            final int val,
            final String value)
    {
        if (val != 0) {
            sb.append(',');
            sb.append(value);
            sb.append('=');
            sb.append(val);
        }
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append(email);
        sb.append("(born-date=");
        FORMATTER.printTo(sb, bornDate);
        appendIf(sb, defaultFlag, DEFAULT);
        appendIf(sb, nativeFlag, NATIVE);
        appendIf(sb, rpopFlag, RPOP);
        appendIf(sb, unsafeFlag, UNSAFE);
        appendIf(sb, validatedFlag, VALIDATED);
        sb.append(')');
    }

    public JsonMap toJsonObject(final ContainerFactory containerFactory) {
        JsonMap result = new JsonMap(containerFactory);
        result.put(ADDRESS, new JsonString(email));
        result.put(BORN_DATE, new JsonLong(bornDate));
        result.put(DEFAULT, JsonBoolean.valueOf(defaultFlag));
        result.put(NATIVE, JsonBoolean.valueOf(nativeFlag));
        result.put(RPOP, JsonBoolean.valueOf(rpopFlag));
        result.put(UNSAFE, JsonBoolean.valueOf(unsafeFlag));
        result.put(VALIDATED, JsonBoolean.valueOf(validatedFlag));
        return result;
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key(ADDRESS);
        writer.value(email);
        writer.key(BORN_DATE);
        writer.value(bornDate);
        writer.key(DEFAULT);
        writer.value(defaultFlag);
        writer.key(NATIVE);
        writer.value(nativeFlag);
        writer.key(RPOP);
        writer.value(rpopFlag);
        writer.key(UNSAFE);
        writer.value(unsafeFlag);
        writer.key(VALIDATED);
        writer.value(validatedFlag);
        writer.endObject();
    }
}

