package ru.yandex.blackbox;

import java.io.IOException;
import java.util.concurrent.Future;
import java.util.function.Function;
import java.util.function.Supplier;

import org.apache.http.Header;
import org.apache.http.HttpHost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;

import ru.yandex.concurrent.FailedFuture;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.json.parser.JsonException;

public class BlackboxClient extends AbstractAsyncClient<BlackboxClient> {
    public static final Function<Exception, RequestErrorType>
        ERROR_CLASSIFIER =
        new Function<Exception, RequestErrorType>() {
            @Override
            public RequestErrorType apply(final Exception e) {
                RequestErrorType type;
                if (e instanceof BlackboxException) {
                    if (e instanceof BlackboxErrorException) {
                        if (((BlackboxErrorException) e).shouldRetry()) {
                            type = RequestErrorType.HTTP;
                        } else {
                            type = RequestErrorType.NON_RETRIABLE;
                        }
                    } else if (
                        e instanceof BlackboxMalformedResponseException)
                    {
                        type = RequestErrorType.HOST_NON_RETRIABLE;
                    } else {
                        /* if (e instanceof BlackboxNotFoundException) */
                        /* if (e instanceof BlackboxOAuthException) */
                        type = RequestErrorType.NON_RETRIABLE;
                    }
                } else if (e instanceof JsonException) {
                    type = RequestErrorType.HOST_NON_RETRIABLE;
                } else {
                    if (e instanceof IOException
                        && e.getCause() instanceof JsonException)
                    {
                        type = RequestErrorType.HOST_NON_RETRIABLE;
                    } else {
                        type = RequestErrorType.ERROR_CLASSIFIER.apply(e);
                    }
                }
                return type;
            }
        };

    private final HttpHost host;

    public BlackboxClient(
        final SharedConnectingIOReactor reactor,
        final ImmutableHttpHostConfig blackboxConfig)
    {
        super(reactor, blackboxConfig, ERROR_CLASSIFIER);
        host = blackboxConfig.host();
    }

    public BlackboxClient(
        final CloseableHttpAsyncClient client,
        final BlackboxClient sample)
    {
        super(client, sample);
        host = sample.host;
    }

    @Override
    public BlackboxClient adjust(final CloseableHttpAsyncClient client) {
        return new BlackboxClient(client, this);
    }

    // CSOFF: ParameterNumber
    public <T> Future<T> execute(
        final BlackboxRequestBase<?> request,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        try {
            BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(
                    request.query().toString());
            for (Header header: request.headers()) {
                producerGenerator.addHeader(header);
            }
            return execute(
                host,
                producerGenerator,
                consumerFactory,
                contextGenerator,
                callback);
        } catch (BadRequestException e) {
            callback.failed(e);
            return new FailedFuture<>(e);
        }
    }
    // CSON: ParameterNumber

    public Future<BlackboxUserinfos> userinfo(
        final BlackboxUserinfoRequest request)
    {
        return userinfo(
            request,
            EmptyFutureCallback.INSTANCE);
    }

    public Future<BlackboxUserinfos> userinfo(
        final BlackboxUserinfoRequest request,
        final FutureCallback<? super BlackboxUserinfos> callback)
    {
        return userinfo(request, httpClientContextGenerator(), callback);
    }

    public Future<BlackboxUserinfos> userinfo(
        final BlackboxUserinfoRequest request,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super BlackboxUserinfos> callback)
    {
        return execute(
            request,
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                new BlackboxUserinfoAsyncResponseConsumerFactory(request)),
            contextGenerator,
            callback
        );
    }

    public Future<BlackboxUserinfo> oauth(final BlackboxOAuthRequest request) {
        return oauth(request, EmptyFutureCallback.INSTANCE);
    }

    public Future<BlackboxUserinfo> oauth(
        final BlackboxOAuthRequest request,
        final FutureCallback<? super BlackboxUserinfo> callback)
    {
        return oauth(request, httpClientContextGenerator(), callback);
    }

    public Future<BlackboxUserinfo> oauth(
        final BlackboxOAuthRequest request,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super BlackboxUserinfo> callback)
    {
        return execute(
            request,
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                new BlackboxOAuthAsyncResponseConsumerFactory(request)),
            contextGenerator,
            callback);
    }

    public Future<BlackboxSessionUserinfo> sessionid(
        final BlackboxSessionidRequest request)
    {
        return sessionid(
            request,
            httpClientContextGenerator(),
            EmptyFutureCallback.INSTANCE);
    }

    public Future<BlackboxSessionUserinfo> sessionid(
        final BlackboxSessionidRequest request,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super BlackboxSessionUserinfo> callback)
    {
        return execute(
            request,
            new StatusCheckAsyncResponseConsumerFactory<>(
                HttpStatusPredicates.OK,
                new BlackboxSessionidAsyncResponseConsumerFactory(request)),
            contextGenerator,
            callback);
    }

    public Future<BlackboxFamilyInfo> familyInfo(
        final BlackboxFamilyInfoRequest request)
    {
        return familyInfo(
            request,
            EmptyFutureCallback.INSTANCE);
    }

    public Future<BlackboxFamilyInfo> familyInfo(
        final BlackboxFamilyInfoRequest request,
        final FutureCallback<? super BlackboxFamilyInfo> callback)
    {
        return familyInfo(request, httpClientContextGenerator(), callback);
    }

    public Future<BlackboxFamilyInfo> familyInfo(
        final BlackboxFamilyInfoRequest request,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super BlackboxFamilyInfo> callback)
    {
        try {
            return execute(
                host,
                new BasicAsyncRequestProducerGenerator(
                    request.query().toString()),
                BlackboxFamilyInfoAsyncResponseConsumerFactory.OK,
                contextGenerator,
                callback);
        } catch (BadRequestException e) {
            callback.failed(e);
            return new FailedFuture<>(e);
        }
    }
}

