package ru.yandex.blackbox;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.parser.string.PositiveLongValidator;

public class BlackboxFamilyInfo
    extends AbstractStringBuilderable
    implements JsonValue
{
    private final String id;
    private final long adminUid;
    private final List<BlackboxFamilyMember> members;

    public BlackboxFamilyInfo(final JsonObject root) throws BlackboxException {
        try {
            JsonMap familyObject = root.get("family").asMap();
            if (familyObject.isEmpty()) {
                throw new JsonException("Empty family object");
            }
            Map.Entry<String, JsonObject> entry =
                familyObject.entrySet().iterator().next();
            id = entry.getKey();
            JsonMap family = entry.getValue().asMap();
            if (family.isEmpty()) {
                throw new BlackboxNotFoundException(id);
            }
            adminUid = family.get("admin_uid", PositiveLongValidator.INSTANCE);
            JsonList users = family.get("users").asListOrNull();
            if (users == null) {
                members = Collections.emptyList();
            } else {
                int size = users.size();
                members = new ArrayList<>(size);
                for (int i = 0; i < size; ++i) {
                    members.add(
                        new BlackboxFamilyMember(users.get(i).asMap()));
                }
            }
        } catch (JsonException e) {
            throw new BlackboxMalformedResponseException(
                "Failed to parse family info from: "
                + JsonType.NORMAL.toString(root),
                e);
        }
    }

    public String id() {
        return id;
    }

    public long adminUid() {
        return adminUid;
    }

    public List<BlackboxFamilyMember> members() {
        return members;
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append("(family_id=");
        sb.append(id);
        sb.append(",admin_uid=");
        sb.append(adminUid);
        sb.append(",users=[");
        int size = members.size();
        for (int i = 0; i < size; ++i) {
            if (i > 0) {
                sb.append(',');
            }
            members.get(i).toStringBuilder(sb);
        }
        sb.append(']');
        sb.append(')');
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key("family_id");
        writer.value(id);
        writer.key("admin_uid");
        writer.value(adminUid);
        writer.key("users");
        writer.startArray();
        int size = members.size();
        for (int i = 0; i < size; ++i) {
            members.get(i).writeValue(writer);
        }
        writer.endArray();
        writer.endObject();
    }

    public JsonMap toJsonObject(final ContainerFactory containerFactory) {
        JsonMap result = new JsonMap(containerFactory, 4);
        result.put("family_id", new JsonString(id));
        result.put("admin_uid", new JsonLong(adminUid));
        int size = members.size();
        JsonList users = new JsonList(containerFactory, size);
        for (int i = 0; i < size; ++i) {
            users.add(members.get(i).toJsonObject(containerFactory));
        }
        result.put("users", users);
        return result;
    }
}
