package ru.yandex.blackbox;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import org.apache.http.Header;
import org.apache.http.message.BasicHeader;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.uri.QueryConstructor;

public abstract class BlackboxRequestBase<T extends BlackboxRequestBase<T>> {
    private final String uri;
    private BlackboxEmailsType emailsType = BlackboxEmailsType.NONE;
    private Set<BlackboxDbfield> dbfields = Collections.emptySet();
    private Set<BlackboxDbfield> requiredDbfields = Collections.emptySet();
    private Set<BlackboxAliasType> aliases = Collections.emptySet();
    private Set<BlackboxAttributeType> attributes = Collections.emptySet();
    private Set<BlackboxPhoneAttributeType> phoneAttributes = Collections.emptySet();
    private String ip = "127.0.0.1";
    private boolean getFamilyInfo = false;
    private List<Header> headers = null;

    protected BlackboxRequestBase(final String method) {
        uri = "/blackbox/?format=json&method=" + method;
    }

    public T emailsType(final BlackboxEmailsType emailsType) {
        this.emailsType = emailsType;
        return self();
    }

    public T dbfields(final BlackboxDbfield dbfield) {
        dbfields = EnumSet.of(dbfield);
        return self();
    }

    public T dbfields(
        final BlackboxDbfield dbfield,
        final BlackboxDbfield... dbfields)
    {
        this.dbfields = EnumSet.of(dbfield, dbfields);
        return self();
    }

    public T dbfields(final Set<BlackboxDbfield> dbfields) {
        this.dbfields = dbfields;
        return self();
    }

    public Set<BlackboxDbfield> requiredDbfields() {
        return requiredDbfields;
    }

    public T requiredDbfields(final BlackboxDbfield dbfield) {
        requiredDbfields = EnumSet.of(dbfield);
        return self();
    }

    public T requiredDbfields(
        final BlackboxDbfield dbfield,
        final BlackboxDbfield... dbfields)
    {
        requiredDbfields = EnumSet.of(dbfield, dbfields);
        return self();
    }

    public T requiredDbfields(final Set<BlackboxDbfield> dbfields) {
        requiredDbfields = dbfields;
        return self();
    }

    public Set<BlackboxAliasType> aliases() {
        return aliases;
    }

    public T aliases(final BlackboxAliasType alias) {
        aliases = EnumSet.of(alias);
        return self();
    }

    public T aliases(
        final BlackboxAliasType alias,
        final BlackboxAliasType... aliases)
    {
        this.aliases = EnumSet.of(alias, aliases);
        return self();
    }

    public T aliases(final Set<BlackboxAliasType> aliases) {
        this.aliases = aliases;
        return self();
    }

    public Set<BlackboxAttributeType> attributes() {
        return attributes;
    }

    public T attributes(final BlackboxAttributeType attribute) {
        attributes = EnumSet.of(attribute);
        return self();
    }

    public T attributes(
        final BlackboxAttributeType attribute,
        final BlackboxAttributeType... attributes)
    {
        this.attributes = EnumSet.of(attribute, attributes);
        return self();
    }

    public T attributes(final Set<BlackboxAttributeType> attributes) {
        this.attributes = attributes;
        return self();
    }

    public T phoneAttributes(final BlackboxPhoneAttributeType attribute) {
        phoneAttributes = EnumSet.of(attribute);
        return self();
    }

    public T phoneAttributes(
            final BlackboxPhoneAttributeType attribute,
            final BlackboxPhoneAttributeType... attributes)
    {
        this.phoneAttributes = EnumSet.of(attribute, attributes);
        return self();
    }

    public T ip(final String ip) {
        this.ip = ip;
        return self();
    }

    public T getFamilyInfo(final boolean getFamilyInfo) {
        this.getFamilyInfo = getFamilyInfo;
        return self();
    }

    public T addHeader(final String name, final String value) {
        return addHeader(new BasicHeader(name, value));
    }

    public T addHeader(final Header header) {
        if (headers == null) {
            headers = new ArrayList<>(1);
        }
        headers.add(header);
        return self();
    }

    public List<Header> headers() {
        if (headers == null) {
            return Collections.emptyList();
        } else {
            return headers;
        }
    }

    protected abstract T self();

    public QueryConstructor query() throws BadRequestException {
        QueryConstructor query = new QueryConstructor(uri);
        query.append("userip", ip);
        Set<BlackboxDbfield> dbfields;
        if (requiredDbfields.isEmpty()) {
            dbfields = this.dbfields;
        } else if (this.dbfields.isEmpty()) {
            dbfields = requiredDbfields;
        } else {
            dbfields = EnumSet.copyOf(this.dbfields);
            dbfields.addAll(requiredDbfields);
        }
        if (!dbfields.isEmpty()) {
            StringBuilder sb = query.sb();
            sb.append("&dbfields=");
            for (BlackboxDbfield dbfield: dbfields) {
                sb.append(dbfield.dbfieldName());
                sb.append(',');
            }
            sb.setLength(sb.length() - 1);
        }
        if (emailsType != BlackboxEmailsType.NONE) {
            StringBuilder sb = query.sb();
            sb.append("&emails=");
            sb.append(emailsType.name().toLowerCase(Locale.ROOT));
        }
        if (!aliases.isEmpty()) {
            StringBuilder sb = query.sb();
            sb.append("&aliases=");
            for (BlackboxAliasType alias: aliases) {
                sb.append(alias.id());
                sb.append(',');
            }
            sb.setLength(sb.length() - 1);
        }
        if (!attributes.isEmpty()) {
            StringBuilder sb = query.sb();
            sb.append("&attributes=");
            for (BlackboxAttributeType attribute: attributes) {
                sb.append(attribute.id());
                sb.append(',');
            }
            sb.setLength(sb.length() - 1);
        }
        if (!phoneAttributes.isEmpty()) {
            StringBuilder sb = query.sb();
            sb.append("&getphones=bound&phone_attributes=");
            for (BlackboxPhoneAttributeType attribute: phoneAttributes) {
                sb.append(attribute.id());
                sb.append(',');
            }
            sb.setLength(sb.length() - 1);
        }
        if (getFamilyInfo) {
            query.sb().append("&get_family_info");
        }

        return query;
    }
}

