package ru.yandex.blackbox;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.annotation.Nonnull;

import ru.yandex.function.AbstractStringBuilderable;
import ru.yandex.function.CorpUidPredicate;
import ru.yandex.function.PddUidPredicate;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

public class BlackboxUserinfo
    extends AbstractStringBuilderable
    implements JsonValue
{
    public static final long CORP_UID_BEGIN = CorpUidPredicate.CORP_UID_BEGIN;
    public static final long CORP_UID_END = CorpUidPredicate.CORP_UID_END;

    public static final long PDD_UID_BEGIN = PddUidPredicate.PDD_UID_BEGIN;
    public static final long PDD_UID_END = PddUidPredicate.PDD_UID_END;

    private static final String HOSTED = "hosted";
    private static final String LITE = "lite";
    private static final String KARMA = "karma";
    private static final String KARMA_STATUS = "karma-status";
    private static final Map<String, BlackboxDbfield> DBFIELD_NAME_TO_DBFIELD;
    private static final Map<String, BlackboxAliasType> ALIAS_NAME_TO_DBFIELD;
    private static final Map<String, BlackboxAttributeType> ATTRIBUTE_NAME_TO_DBFIELD;
    private static final Map<String, BlackboxPhoneAttributeType> PHONE_ATTRIBUTE_NAME_TO_DBFIELD;

    static {
        DBFIELD_NAME_TO_DBFIELD = new HashMap<>(BlackboxDbfield.values().length << 2);
        for (BlackboxDbfield dbfield: BlackboxDbfield.values()) {
            DBFIELD_NAME_TO_DBFIELD.put(dbfield.dbfieldName(), dbfield);
        }
        ALIAS_NAME_TO_DBFIELD = new HashMap<>(BlackboxAliasType.values().length << 2);
        for (BlackboxAliasType alias: BlackboxAliasType.values()) {
            ALIAS_NAME_TO_DBFIELD.put(Integer.toString(alias.id()), alias);
        }
        ATTRIBUTE_NAME_TO_DBFIELD = new HashMap<>(BlackboxAttributeType.values().length << 2);
        for (BlackboxAttributeType attribute: BlackboxAttributeType.values()) {
            ATTRIBUTE_NAME_TO_DBFIELD.put(
                Integer.toString(attribute.id()),
                attribute);
        }
        PHONE_ATTRIBUTE_NAME_TO_DBFIELD = new HashMap<>(BlackboxPhoneAttributeType.values().length << 2);
        for (BlackboxPhoneAttributeType attribute : BlackboxPhoneAttributeType.values()) {
            PHONE_ATTRIBUTE_NAME_TO_DBFIELD.put(
                Integer.toString(attribute.id()),
                attribute);
        }
    }

    @Nonnull
    private final JsonMap source;

    private final long uid;
    private final boolean hosted;
    private final boolean lite;
    private final String login;
    private final int karma;
    private final int karmaStatus;
    private final List<BlackboxAddress> addressList;
    private final Map<BlackboxDbfield, String> dbfields;
    private final Map<BlackboxAliasType, JsonObject> aliases;
    private final Map<BlackboxAttributeType, JsonObject> attributes;
    private final Map<String, Map<BlackboxPhoneAttributeType, String>> phoneAttributes;
    private final BlackboxUserinfoFamilyInfo familyInfo;

    public BlackboxUserinfo(
        final BlackboxRequestBase<?> request,
        final JsonMap map)
        throws BlackboxException
    {
        this.source = map;
        try {
            JsonMap uidMap = map.getMap("uid");
            if (uidMap.isEmpty()) {
                // Will never happen for oauth request, used for userinfo ones
                throw new BlackboxNotFoundException(map.getString("id", ""));
            }

            uid = uidMap.getLong("value");
            hosted = uidMap.getBoolean(HOSTED);
            lite = uidMap.getBoolean(LITE);

            login = map.getString("login");

            karma = map.getMap("karma").getInt("value");
            karmaStatus = map.getMap("karma_status").getInt("value");

            JsonList list = map.getListOrNull("address-list");
            if (list == null) {
                addressList = Collections.emptyList();
            } else {
                addressList = new ArrayList<>(list.size());
                for (JsonObject address: list) {
                    addressList.add(new BlackboxAddress(address.asMap()));
                }
            }

            JsonMap fields = map.getMapOrNull("dbfields");
            if (fields == null) {
                dbfields = Collections.emptyMap();
            } else {
                dbfields = new EnumMap<>(BlackboxDbfield.class);
                for (Map.Entry<String, JsonObject> entry: fields.entrySet()) {
                    String key = entry.getKey();
                    BlackboxDbfield dbfield = DBFIELD_NAME_TO_DBFIELD.get(key);
                    if (dbfield == null) {
                        throw new BlackboxMalformedResponseException(
                            "Unknown dbfield encountered: " + key
                            + " in " + JsonType.NORMAL.toString(map));
                    }
                    dbfield.addToMap(dbfields, entry.getValue());
                }
            }
            for (BlackboxDbfield dbfield: request.requiredDbfields()) {
                if (!dbfields.containsKey(dbfield)) {
                    throw new BlackboxFieldMissingException(dbfield, map);
                }
            }

            JsonMap aliases = map.getMapOrNull("aliases");
            if (aliases == null) {
                this.aliases = Collections.emptyMap();
            } else {
                this.aliases = new EnumMap<>(BlackboxAliasType.class);
                for (Map.Entry<String, JsonObject> entry: aliases.entrySet()) {
                    String key = entry.getKey();
                    BlackboxAliasType alias = ALIAS_NAME_TO_DBFIELD.get(key);
                    if (alias == null) {
                        throw new BlackboxMalformedResponseException(
                            "Unknown alias encountered: " + key
                            + " in " + JsonType.NORMAL.toString(map));
                    }
                    alias.addToMap(this.aliases, entry.getValue());
                }
            }

            JsonMap attributes = map.getMapOrNull("attributes");
            if (attributes == null) {
                this.attributes = Collections.emptyMap();
            } else {
                this.attributes = new EnumMap<>(BlackboxAttributeType.class);
                for (Map.Entry<String, JsonObject> entry
                    : attributes.entrySet())
                {
                    String key = entry.getKey();
                    BlackboxAttributeType attribute =
                        ATTRIBUTE_NAME_TO_DBFIELD.get(key);
                    if (attribute == null) {
                        throw new BlackboxMalformedResponseException(
                            "Unknown attribute encountered: " + key
                            + " in " + JsonType.NORMAL.toString(map));
                    }
                    attribute.addToMap(this.attributes, entry.getValue());
                }
            }

            JsonList phones = map.getListOrNull("phones");
            if (phones == null) {
                this.phoneAttributes = Collections.emptyMap();
            } else {
                this.phoneAttributes = new HashMap<>();
                for (JsonObject phoneEntry : phones) {
                    final String id = phoneEntry.get("id").asString();
                    final JsonMap rawAttributes = phoneEntry.get("attributes").asMap();

                    final Map<BlackboxPhoneAttributeType, String> attributesMap = phoneAttributes.computeIfAbsent(
                            id,
                            ignored -> new EnumMap<>(BlackboxPhoneAttributeType.class));

                    for (Map.Entry<String, JsonObject> attrEntry : rawAttributes.entrySet()) {
                        String key = attrEntry.getKey();
                        BlackboxPhoneAttributeType attribute =
                            PHONE_ATTRIBUTE_NAME_TO_DBFIELD.get(key);
                        if (attribute == null) {
                            throw new BlackboxMalformedResponseException(
                                "Unknown attribute encountered: " + key
                                    + " in " + JsonType.NORMAL.toString(map));
                        }
                        attribute.addToMap(attributesMap, attrEntry.getValue());
                    }
                }
            }

            JsonMap familyInfo = map.getMapOrNull("family_info");
            if (familyInfo == null || familyInfo.isEmpty()) {
                this.familyInfo = null;
            } else {
                this.familyInfo = new BlackboxUserinfoFamilyInfo(familyInfo);
            }
        } catch (JsonException e) {
            throw new BlackboxMalformedResponseException(
                "Failed to parse user info from: "
                + JsonType.NORMAL.toString(map),
                e);
        }
    }

    public BlackboxUserinfo(final long uid) {
        source = JsonMap.EMPTY;
        this.uid = uid;
        hosted = false;
        lite = false;
        login = "unresolved";
        karma = 0;
        karmaStatus = 0;
        dbfields = new HashMap<>();
        addressList = new ArrayList<>();
        aliases = new HashMap<>();
        attributes = new HashMap<>();
        phoneAttributes = new HashMap<>();
        familyInfo = null;
    }

    @Nonnull
    public JsonMap source() {
        return source;
    }

    public String login() {
        return login;
    }

    public int karma() {
        return karma;
    }

    public int karmaStatus() {
        return karmaStatus;
    }

    public long uid() {
        return uid;
    }

    public boolean hosted() {
        return hosted;
    }

    public boolean lite() {
        return lite;
    }

    public List<BlackboxAddress> addressList() {
        return addressList;
    }

    public Map<BlackboxDbfield, String> dbfields() {
        return dbfields;
    }

    public Map<BlackboxAliasType, JsonObject> aliases() {
        return aliases;
    }

    public Map<BlackboxAttributeType, JsonObject> attributes() {
        return attributes;
    }

    public Map<String, Map<BlackboxPhoneAttributeType, String>>
        phoneAttributes()
    {
        return phoneAttributes;
    }

    public BlackboxUserinfoFamilyInfo familyInfo() {
        return familyInfo;
    }

    public boolean corp() {
        return corp(uid);
    }

    public static boolean corp(final long uid) {
        return CorpUidPredicate.INSTANCE.test(uid);
    }

    public boolean pdd() {
        return pdd(uid);
    }

    public static boolean pdd(final long uid) {
        return PddUidPredicate.INSTANCE.test(uid);
    }

    public boolean hasSecurePhone() {
        for (Map<BlackboxPhoneAttributeType, String> phoneAttributes
            : this.phoneAttributes.values())
        {
            boolean hasSecurePhone =
                "1".equals(
                    phoneAttributes.get(
                        BlackboxPhoneAttributeType.IS_SECURED));
            if (hasSecurePhone) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void toStringBuilder(final StringBuilder sb) {
        sb.append(login);
        sb.append("(uid=");
        sb.append(uid);
        BlackboxAddress.appendIf(sb, karma, KARMA);
        BlackboxAddress.appendIf(sb, karmaStatus, KARMA_STATUS);
        BlackboxAddress.appendIf(sb, hosted, HOSTED);
        BlackboxAddress.appendIf(sb, lite, LITE);
        if (!addressList.isEmpty()) {
            sb.append(',');
            sb.append('[');
            addressList.get(0).toStringBuilder(sb);
            for (int i = 1; i < addressList.size(); ++i) {
                sb.append(',');
                addressList.get(i).toStringBuilder(sb);
            }
            sb.append(']');
        }
        if (!dbfields.isEmpty()) {
            sb.append(',');
            sb.append('{');
            Iterator<Map.Entry<BlackboxDbfield, String>> iter =
                dbfields.entrySet().iterator();
            Map.Entry<BlackboxDbfield, String> entry = iter.next();
            sb.append(entry.getKey().dbfieldName());
            sb.append('=');
            sb.append(entry.getValue());
            while (iter.hasNext()) {
                entry = iter.next();
                sb.append(',');
                sb.append(entry.getKey().dbfieldName());
                sb.append('=');
                sb.append(entry.getValue());
            }
            sb.append('}');
        }
        if (!aliases.isEmpty()) {
            sb.append(",aliases={");
            Iterator<Map.Entry<BlackboxAliasType, JsonObject>> iter =
                aliases.entrySet().iterator();
            Map.Entry<BlackboxAliasType, JsonObject> entry = iter.next();
            sb.append(entry.getKey().name());
            sb.append('=');
            JsonType.NORMAL.toStringBuilder(sb, entry.getValue());
            while (iter.hasNext()) {
                entry = iter.next();
                sb.append(',');
                sb.append(entry.getKey().name());
                sb.append('=');
                JsonType.NORMAL.toStringBuilder(sb, entry.getValue());
            }
            sb.append('}');
        }
        if (!attributes.isEmpty()) {
            sb.append(",attributes={");
            Iterator<Map.Entry<BlackboxAttributeType, JsonObject>> iter =
                attributes.entrySet().iterator();
            Map.Entry<BlackboxAttributeType, JsonObject> entry = iter.next();
            sb.append(entry.getKey().name());
            sb.append('=');
            sb.append(entry.getValue());
            while (iter.hasNext()) {
                entry = iter.next();
                sb.append(',');
                sb.append(entry.getKey().name());
                sb.append('=');
                JsonType.NORMAL.toStringBuilder(sb, entry.getValue());
            }
            sb.append('}');
        }
        if(!phoneAttributes.isEmpty()) {
            sb.append(",phones={");
            Iterator<Map.Entry<String, Map<BlackboxPhoneAttributeType, String>>>
                iter = phoneAttributes.entrySet().iterator();
            Map.Entry<String, Map<BlackboxPhoneAttributeType, String>> entry =
                iter.next();

            sb.append(entry.getKey());
            sb.append('=');
            toStringBuilder(sb, entry.getValue());
            while (iter.hasNext()) {
                entry = iter.next();
                sb.append(',');
                sb.append(entry.getKey());
                sb.append('=');
                toStringBuilder(sb, entry.getValue());
            }
            sb.append('}');
        }
        if (familyInfo != null) {
            sb.append(",family_info={family_id=");
            sb.append(familyInfo.id());
            sb.append(",admin_uid=");
            sb.append(familyInfo.adminUid());
            sb.append('}');
        }
        sb.append(')');
    }

    public JsonMap toJsonObject(final ContainerFactory containerFactory) {
        JsonMap result = new JsonMap(containerFactory);
        result.put("uid", new JsonLong(uid));
        result.put("hosted", JsonBoolean.valueOf(hosted));
        result.put("lite", JsonBoolean.valueOf(lite));
        result.put("login", new JsonString(login));
        result.put(KARMA, new JsonLong(karma));
        result.put(KARMA_STATUS, new JsonLong(karmaStatus));

        int size = addressList.size();
        JsonList addressList = new JsonList(containerFactory, size);
        for (int i = 0; i < size; ++i) {
            addressList.add(
                this.addressList.get(i).toJsonObject(containerFactory));
        }
        result.put("address-list", addressList);

        JsonMap dbfields = new JsonMap(containerFactory);
        for (Map.Entry<BlackboxDbfield, String> entry
            : this.dbfields.entrySet())
        {
            dbfields.put(
                entry.getKey().dbfieldName(),
                new JsonString(entry.getValue()));
        }
        result.put("dbfields", dbfields);

        JsonMap aliases = new JsonMap(containerFactory);
        for (Map.Entry<BlackboxAliasType, JsonObject> entry
            : this.aliases.entrySet())
        {
            aliases.put(
                entry.getKey().name().toLowerCase(Locale.ROOT),
                entry.getValue());
        }
        result.put("aliases", aliases);

        JsonMap attributes = new JsonMap(containerFactory);
        for (Map.Entry<BlackboxAttributeType, JsonObject> entry
            : this.attributes.entrySet())
        {
            attributes.put(
                entry.getKey().name().toLowerCase(Locale.ROOT),
                entry.getValue());
        }
        result.put("attributes", attributes);

        {
            JsonMap phoneAttributes = new JsonMap(containerFactory);

            for(Map.Entry<String, Map<BlackboxPhoneAttributeType, String>> entry : this.phoneAttributes.entrySet()) {
                JsonMap idAttributes = new JsonMap(containerFactory);

                for(Map.Entry<BlackboxPhoneAttributeType, String> idEntry : entry.getValue().entrySet()) {
                    idAttributes.put(
                            idEntry.getKey().name().toLowerCase(Locale.ROOT),
                            new JsonString(idEntry.getValue()));
                }

                phoneAttributes.put(entry.getKey(), idAttributes);
            }

            result.put("phones", phoneAttributes);
        }

        if (this.familyInfo != null) {
            JsonMap familyInfo = new JsonMap(containerFactory, 4);
            familyInfo.put("family_id", new JsonString(this.familyInfo.id()));
            familyInfo.put(
                "admin_uid",
                new JsonLong(this.familyInfo.adminUid()));
            result.put("family_info", familyInfo);
        }

        return result;
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key("uid");
        writer.value(uid);
        writer.key("hosted");
        writer.value(hosted);
        writer.key("lite");
        writer.value(lite);
        writer.key("login");
        writer.value(login);
        writer.key(KARMA);
        writer.value(karma);
        writer.key(KARMA_STATUS);
        writer.value(karmaStatus);

        writer.key("address-list");
        writer.startArray();
        for (BlackboxAddress address: addressList) {
            writer.value(address);
        }
        writer.endArray();

        writer.key("dbfields");
        writer.startObject();
        for (Map.Entry<BlackboxDbfield, String> entry: dbfields.entrySet()) {
            writer.key(entry.getKey().dbfieldName());
            writer.value(entry.getValue());
        }
        writer.endObject();

        writer.key("aliases");
        writer.startObject();
        for (Map.Entry<BlackboxAliasType, JsonObject> entry
            : aliases.entrySet())
        {
            writer.key(entry.getKey().name().toLowerCase(Locale.ROOT));
            entry.getValue().writeValue(writer);
        }
        writer.endObject();

        writer.key("attributes");
        writer.startObject();
        for (Map.Entry<BlackboxAttributeType, JsonObject> entry
            : attributes.entrySet())
        {
            writer.key(entry.getKey().name().toLowerCase(Locale.ROOT));
            entry.getValue().writeValue(writer);
        }
        writer.endObject();

        writer.key("phones");
        writer.startObject();
        for(Map.Entry<String, Map<BlackboxPhoneAttributeType, String>> entry : this.phoneAttributes.entrySet()) {
            writer.key(entry.getKey());
            writer.startObject();

            for(Map.Entry<BlackboxPhoneAttributeType, String> idEntry : entry.getValue().entrySet()) {
                writer.key(idEntry.getKey().name().toLowerCase(Locale.ROOT));
                writer.value(entry.getValue());
            }

            writer.endObject();
        }
        writer.endObject();

        if (familyInfo != null) {
            writer.key("family_info");
            writer.startObject();
            writer.key("family_id");
            writer.value(familyInfo.id());
            writer.key("admin_uid");
            writer.value(familyInfo.adminUid());
            writer.endObject();
        }

        writer.endObject();
    }

    private static void toStringBuilder(final StringBuilder sb, Map<BlackboxPhoneAttributeType, String> attributes) {
        sb.append('{');

        Iterator<Map.Entry<BlackboxPhoneAttributeType, String>> iter =
                attributes.entrySet().iterator();

        Map.Entry<BlackboxPhoneAttributeType, String> entry = iter.next();

        sb.append(entry.getKey());
        sb.append('=');
        sb.append(entry.getValue());
        while (iter.hasNext()) {
            entry = iter.next();
            sb.append(',');
            sb.append(entry.getKey());
            sb.append('=');
            sb.append(entry.getValue());
        }

        sb.append('}');
    }
}

