package ru.yandex.util.storage.cocaine;

import java.io.IOException;
import java.util.logging.Logger;

import org.apache.http.HttpException;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineServiceException;
import ru.yandex.client.cocaine.unistorage.UnistorageInputStream;
import ru.yandex.client.cocaine.unistorage.UnistorageService;
import ru.yandex.client.cocaine.unistorage.UnistorageSession;
import ru.yandex.client.cocaine.worker.CocaineWorkerSession;
import ru.yandex.client.cocaine.worker.http.AbstractCocaineHttpRequestHandler;
import ru.yandex.client.cocaine.worker.http.CocaineServiceExceptionConverter;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.UnsupportedMediaTypeException;
import ru.yandex.util.storage.DataExtractor;
import ru.yandex.util.storage.ImmutableDataExtractorConfig;
import ru.yandex.util.storage.StorageData;

public class CocaineStorageClient {
    private static final String MS = " ms";

    private final UnistorageService unistorage;
    private final DataExtractor dataExtractor;
    private final ImmutableDataExtractorConfig dataExtractorConfig;

    public CocaineStorageClient(
        final UnistorageService unistorage,
        final DataExtractor dataExtractor,
        final ImmutableDataExtractorConfig dataExtractorConfig)
    {
        this.unistorage = unistorage;
        this.dataExtractor = dataExtractor;
        this.dataExtractorConfig = dataExtractorConfig;
    }

    public StorageData sendStorageRequest(
        final String stid,
        final CocaineWorkerSession session,
        final Logger logger)
        throws HttpException
    {
        logger.info("Requesting stid: " + stid + " from " + unistorage);
        long start = System.currentTimeMillis();
        try (UnistorageSession unistorageSession =
                unistorage.read(stid, session.headers()))
        {
            unistorageSession.requestMeta();
            long contentLength = unistorageSession.get().asMeta().size();
            long end = System.currentTimeMillis();
            logger.info(
                "Got file size " + contentLength + " on " + session
                + " in " + (end - start) + MS);
            try (UnistorageInputStream in =
                    new UnistorageInputStream(unistorageSession))
            {
                StorageData storageData =
                    dataExtractor.extractData(
                        in,
                        contentLength,
                        dataExtractorConfig);
                if (storageData == null) {
                    throw new UnsupportedMediaTypeException(
                        "No data extracted");
                }
                StringBuilder sb = new StringBuilder();
                storageData.toStringBuilder(sb);
                sb.append(" extracted in ");
                sb.append(System.currentTimeMillis() - end);
                sb.append(MS);
                logger.info(new String(sb));
                return storageData;
            }
        } catch (CocaineServiceException e) {
            throw CocaineServiceExceptionConverter.INSTANCE.apply(e);
        } catch (IOException e) {
            AbstractCocaineHttpRequestHandler.unwrapCocaineServiceException(e);
            throw new BadGatewayException(e);
        } catch (CocaineException | InterruptedException e) {
            throw new BadGatewayException(e);
        }
    }
}

