package ru.yandex.cokemulator.srw;

import java.io.IOException;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.logging.Logger;

import cocaine.hpack.HeaderField;
import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.worker.CocaineWorkerConfig;
import ru.yandex.client.cocaine.worker.CocaineWorkerSession;
import ru.yandex.client.cocaine.worker.http.AbstractCocaineHttpRequestHandler;
import ru.yandex.client.cocaine.worker.http.CocaineDummyHttpRequestHandler;
import ru.yandex.client.cocaine.worker.http.CocaineExecutorHttpRequestHandler;
import ru.yandex.client.cocaine.worker.http.CocaineHttpEventHandler;
import ru.yandex.client.cocaine.worker.http.unistorage.UnistorageHttpService;
import ru.yandex.concurrent.SemaphoreLockAdapter;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.function.GenericNonThrowingCloseableAdapter;
import ru.yandex.http.util.ByteArrayEntityFactory;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.UnsupportedMediaTypeException;
import ru.yandex.jniwrapper.JniWrapper;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.jniwrapper.JniWrapperUnprocessableInputException;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.ScanningCgiParams;
import ru.yandex.util.storage.C2CDataExtractor;
import ru.yandex.util.storage.StorageData;
import ru.yandex.util.storage.cocaine.CocaineStorageClient;

public class CokemulatorSrw extends UnistorageHttpService {
    private static final String STID = "stid";

    private final ImmutableCokemulatorSrwConfig cokemulatorConfig;
    private final CocaineStorageClient storageClient;
    private final Lock processingLock;
    private final JniWrapper jniWrapper;

    public CokemulatorSrw(
        final ImmutableCokemulatorSrwConfig cokemulatorConfig,
        final CocaineWorkerConfig workerConfig)
        throws CocaineException,
            IOException,
            InterruptedException,
            JniWrapperException
    {
        super(cokemulatorConfig, workerConfig, "CokemulatorSrw");
        this.cokemulatorConfig = cokemulatorConfig;
        storageClient = new CocaineStorageClient(
            unistorage,
            cokemulatorConfig.dataType().dataExtractor(),
            cokemulatorConfig.dataExtractorConfig());
        processingLock = SemaphoreLockAdapter.create(
            cokemulatorConfig.concurrency(),
            cokemulatorConfig.workers());
        try (GenericAutoCloseableHolder<
                IOException,
                GenericAutoCloseableChain<IOException>> chain =
                    new GenericAutoCloseableHolder<>(this.chain))
        {
            jniWrapper = JniWrapper.create(
                cokemulatorConfig.jniWrapperConfig(),
                new ThreadFactoryConfig("Jni-")
                    .group(Thread.currentThread().getThreadGroup())
                    .daemon(true)
                    .uncaughtExceptionHandler(this));
            chain.get().add(
                new GenericNonThrowingCloseableAdapter<>(jniWrapper));
            handlerRegistry.register(
                "process",
                new CocaineHttpEventHandler(
                    new CocaineExecutorHttpRequestHandler(
                        executor,
                        new ProcessHandler()),
                    cokemulatorConfig.readTimeout()));
            handlerRegistry.register(
                "get-text",
                new CocaineHttpEventHandler(
                    new CocaineExecutorHttpRequestHandler(
                        executor,
                        new GetTextHandler()),
                    cokemulatorConfig.readTimeout()));
            chain.release();
        }
    }

    public static void main(final String... args)
        throws CocaineException,
            ConfigException,
            IOException,
            InterruptedException,
            JniWrapperException
    {
        IniConfig config = new IniConfig(Paths.get(args[0]));
        CokemulatorSrw server = new CokemulatorSrw(
            new ImmutableCokemulatorSrwConfig(
                new CokemulatorSrwConfigBuilder(config)),
            new CocaineWorkerConfig(args));
        config.checkUnusedKeys();
        server.start();
    }

    private class ProcessHandler extends AbstractCocaineHttpRequestHandler {
        ProcessHandler() {
            super(logging, cokemulatorConfig.minRequestTime());
        }

        // CSOFF: ParameterNumber
        @Override
        protected void handle(
            final HttpRequest request,
            final List<HeaderField> headers,
            final CocaineWorkerSession session,
            final Logger logger)
            throws HttpException
        {
            String stid = new ScanningCgiParams(request).get(
                STID,
                NonEmptyValidator.INSTANCE);
            StorageData storageData =
                storageClient.sendStorageRequest(stid, session, logger);
            String processingResult;
            long processingTime;
            try {
                processingLock.lock();
                try {
                    long start = System.currentTimeMillis();
                    processingResult = storageData.processWith(
                        request.getRequestLine().getUri(),
                        jniWrapper);
                    processingTime = System.currentTimeMillis() - start;
                } finally {
                    processingLock.unlock();
                }
            } catch (JniWrapperUnprocessableInputException e) {
                throw new UnsupportedMediaTypeException(e);
            } catch (JniWrapperException e) {
                throw new ServiceUnavailableException(e);
            }
            logger.info(
                "Processing time: " + processingTime
                + " ms, response size: " + processingResult.length());
            new CocaineDummyHttpRequestHandler(
                HttpStatus.SC_OK,
                new StringEntity(
                    processingResult,
                    cokemulatorConfig.contentType().withCharset(
                        CharsetUtils.acceptedCharset(request))))
                .handle(request, headers, session);
        }
        // CSON: ParameterNumber
    }

    private class GetTextHandler extends AbstractCocaineHttpRequestHandler {
        GetTextHandler() {
            super(logging, 0L);
        }

        // CSOFF: ParameterNumber
        @Override
        protected void handle(
            final HttpRequest request,
            final List<HeaderField> headers,
            final CocaineWorkerSession session,
            final Logger logger)
            throws HttpException
        {
            String stid = new ScanningCgiParams(request).get(
                STID,
                NonEmptyValidator.INSTANCE);
            StorageData storageData =
                storageClient.sendStorageRequest(stid, session, logger);
            List<Header> responseHeaders;
            String metainfo = storageData.metainfo();
            if (metainfo == null) {
                responseHeaders = Collections.emptyList();
            } else {
                responseHeaders =
                    Collections.singletonList(
                        new BasicHeader(C2CDataExtractor.METAINFO, metainfo));
            }
            new CocaineDummyHttpRequestHandler(
                HttpStatus.SC_OK,
                storageData.processWith(ByteArrayEntityFactory.INSTANCE),
                responseHeaders)
                .handle(request, headers, session);
        }
        // CSON: ParameterNumber
    }
}

