package ru.yandex.client.cocaine;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.SocketTimeoutException;

import cocaine.message.Message;
import cocaine.msgpack.InvocationRequestTemplate;
import cocaine.msgpack.MessageTemplate;
import cocaine.service.invocation.InvocationRequest;
import org.msgpack.MessagePack;
import org.msgpack.unpacker.Unpacker;

import ru.yandex.function.GenericAutoCloseable;

public abstract class AbstractCocaineClient
    implements CocaineClient, GenericAutoCloseable<IOException>
{
    private static final MessagePack MESSAGE_PACK = new MessagePack();
    private static final InvocationRequestTemplate REQUEST_TEMPLATE =
        new InvocationRequestTemplate();

    protected final UniSocket socket;
    protected final CocaineClientCallback callback;
    protected volatile boolean closed = false;
    private final Thread eventThread;

    protected AbstractCocaineClient(final CocaineClientContext clientContext) {
        socket = clientContext.socket();
        callback = clientContext.callback();
        eventThread =
            clientContext.eventThreadFactory().newThread(() -> eventLoop());
    }

    public static void serializeRequest(
        final InvocationRequest request,
        final OutputStream out)
        throws IOException
    {
        REQUEST_TEMPLATE.write(MESSAGE_PACK.createPacker(out), request);
    }

    @Override
    public void start() {
        eventThread.start();
    }

    @Override
    @SuppressWarnings("ThreadJoinLoop")
    public void close() throws IOException {
        closed = true;
        eventThread.interrupt();
        socket.close();
        try {
            eventThread.join();
        } catch (InterruptedException e) {
            // Keep calm and leave
        }
    }

    private void eventLoop() {
        BufferedInputStream in = new BufferedInputStream(socket.in());
        Unpacker unpacker = MESSAGE_PACK.createUnpacker(in);
        MessageTemplate template = new MessageTemplate();
        while (!closed) {
            Message message;
            try {
                in.mark(1);
                try {
                    in.read();
                } catch (SocketTimeoutException e) {
                    continue;
                } finally {
                    in.reset();
                }
                message = template.read(unpacker, null, true);
            } catch (IOException e) {
                if (!closed) {
                    callback.readFailed(e);
                }
                continue;
            }
            callback.messageReceived(message);
        }
    }

    @Override
    public String toString() {
        return "client-endpoint:" + socket;
    }
}

