package ru.yandex.client.cocaine;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cocaine.hpack.HeaderField;
import cocaine.message.Message;
import cocaine.service.invocation.InvocationRequest;

import ru.yandex.client.cocaine.protocol.CocaineMethodApi;
import ru.yandex.function.GenericAutoCloseable;

public class CocaineService
    implements CocaineClientCallback, GenericAutoCloseable<IOException>
{
    protected final Map<Long, CocaineSession> sessions = new HashMap<>();
    protected final IOErrorHandler ioErrorHandler;
    protected final Map<Integer, CocaineMethodApi> methods;
    protected final CocaineClient client;
    protected long maxSessionId = 0L;

    public CocaineService(
        final IOErrorHandler ioErrorHandler,
        final CocaineServiceContext context)
    {
        this.ioErrorHandler = ioErrorHandler;
        methods = context.methods();
        client = context.clientFactory().create(context.socket(), this);
    }

    public void sendRequest(final InvocationRequest request) {
        client.sendRequest(request);
    }

    public void start() {
        client.start();
    }

    public void removeSession(final Long sessionId) {
        synchronized (sessions) {
            sessions.remove(sessionId);
        }
    }

    @Override
    public void close() throws IOException {
        client.close();
    }

    @Override
    public void readFailed(final IOException e) {
        ioErrorHandler.readFailed(e);
    }

    @Override
    public void writeFailed(final IOException e) {
        ioErrorHandler.writeFailed(e);
    }

    protected void unknownSessionId(final Message message) {
    }

    protected void messageProcessingFailed(
        final CocaineSession session,
        final Message message,
        final CocaineException e)
    {
        session.failed(e);
        session.close();
    }

    protected CocaineSession sessionFor(final Message message) {
        Long sessionId = message.getSession();
        synchronized (sessions) {
            return sessions.get(sessionId);
        }
    }

    @Override
    public void messageReceived(final Message message) {
        CocaineSession session = sessionFor(message);
        if (session == null) {
            unknownSessionId(message);
        } else {
            try {
                session.messageReceived(message);
            } catch (CocaineException e) {
                messageProcessingFailed(session, message, e);
            }
        }
    }

    public <T extends CocaineSession> T createSession(
        final CocaineSessionFactory<T> sessionFactory,
        final int methodId,
        final List<Object> args)
        throws CocaineException, IOException
    {
        return createSession(
            sessionFactory,
            methodId,
            args,
            Collections.emptyList());
    }

    // CSOFF: ParameterNumber
    public <T extends CocaineSession> T createSession(
        final CocaineSessionFactory<T> sessionFactory,
        final int methodId,
        final List<Object> args,
        final List<HeaderField> headers)
        throws CocaineException, IOException
    {
        CocaineMethodApi methodApi = methods.get(methodId);
        if (methodApi == null) {
            throw new CocaineProtocolException(
                "Unknown method id: " + methodId);
        }
        synchronized (this) {
            long sessionId = ++maxSessionId;
            T session = sessionFactory.createSession(
                new CocaineSessionContext(sessionId, this, methodApi));
            Long sessionIdBoxed = sessionId;
            synchronized (sessions) {
                sessions.put(sessionIdBoxed, session);
            }
            sendRequest(
                new InvocationRequest(methodId, sessionId, args, headers));
            return session;
        }
    }
    // CSON: ParameterNumber

    @Override
    public String toString() {
        return client.toString();
    }
}

