package ru.yandex.client.cocaine;

import java.io.IOException;
import java.util.Collections;
import java.util.List;

import cocaine.hpack.HeaderField;
import cocaine.message.Message;
import cocaine.service.invocation.InvocationRequest;
import org.msgpack.type.Value;

import ru.yandex.client.cocaine.protocol.CocaineMethodApi;
import ru.yandex.client.cocaine.protocol.CocaineProtocol;
import ru.yandex.client.cocaine.protocol.fsm.FsmState;
import ru.yandex.client.cocaine.protocol.fsm.FsmTransition;
import ru.yandex.function.GenericAutoCloseable;

public abstract class CocaineSession
    implements GenericAutoCloseable<RuntimeException>
{
    protected volatile boolean readable = true;
    protected volatile boolean writable = true;
    private final long id;
    private final CocaineService cocaineService;
    private final CocaineProtocol receiveProtocol;
    private FsmState transmitState;
    private FsmState receiveState;

    protected CocaineSession(
        final CocaineSessionContext cocaineSessionContext)
    {
        id = cocaineSessionContext.id();
        cocaineService = cocaineSessionContext.service();
        CocaineMethodApi methodApi = cocaineSessionContext.methodApi();
        receiveProtocol = methodApi.receiveProtocol();
        transmitState = methodApi.initialTransmitState();
        receiveState = methodApi.initialReceiveState();
    }

    public FsmState transmitState() {
        return transmitState;
    }

    protected abstract void failed(final CocaineException e);

    protected abstract void payloadReceived(
        final String messageType,
        final Value payload,
        final List<HeaderField> headers)
        throws CocaineException;

    protected void implShutdownInput() {
    }

    protected void implShutdownOutput() {
    }

    private void shutdownInput() {
        if (readable) {
            readable = false;
            implShutdownInput();
        }
    }

    private void shutdownOutput() {
        if (writable) {
            writable = false;
            implShutdownOutput();
        }
    }

    @Override
    public void close() {
        shutdownInput();
        shutdownOutput();
        cocaineService.removeSession(id);
    }

    public void messageReceived(final Message message)
        throws CocaineException
    {
        if (!readable) {
            return;
        }
        FsmTransition transition = receiveState.transition(message.getType());
        if (transition == null) {
            throw new UnexpectedCocaineMessageException(message);
        } else {
            String messageType = transition.messageName();
            payloadReceived(
                messageType,
                receiveProtocol.handle(messageType, message.getPayload()),
                message.getHeaders());
            receiveState = transition.state();
            if (receiveState.finalState()) {
                shutdownInput();
            }
        }
    }

    public void sendMessage(final int messageId, final List<Object> args)
        throws IOException
    {
        sendMessage(messageId, args, Collections.emptyList());
    }

    public void sendMessage(
        final int messageId,
        final List<Object> args,
        final List<HeaderField> headers)
        throws IOException
    {
        if (!writable) {
            throw new IOException("Session output is closed");
        }
        FsmTransition transition = transmitState.transition(messageId);
        if (transition == null) {
            throw new IOException("Unknown message id: " + messageId);
        } else {
            cocaineService.sendRequest(
                new InvocationRequest(messageId, id, args, headers));
            transmitState = transition.state();
            if (transmitState.finalState()) {
                shutdownOutput();
            }
        }
    }

    @Override
    public String toString() {
        return "session-id:" + id;
    }
}

