package ru.yandex.client.cocaine;

import java.io.IOException;
import java.net.SocketTimeoutException;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

import cocaine.hpack.HeaderField;
import org.msgpack.type.Value;

public class PipedCocaineSession<T> extends CocaineSession {
    private static final Node NULL_NODE = new Node() {
        @Override
        public <T> T deserialize(
            final CocainePayloadDeserializer<? extends T> deserializer)
        {
            return null;
        }
    };

    private final BlockingQueue<Node> queue = new LinkedBlockingQueue<>();
    private final long readTimeout;
    private final CocainePayloadDeserializer<? extends T> deserializer;

    public PipedCocaineSession(
        final CocaineSessionContext cocaineSessionContext,
        final long readTimeout,
        final CocainePayloadDeserializer<? extends T> deserializer)
    {
        super(cocaineSessionContext);
        this.readTimeout = readTimeout;
        this.deserializer = deserializer;
    }

    @Override
    protected void failed(final CocaineException e) {
        queue.add(new ErrorNode(e));
    }

    @Override
    protected void payloadReceived(
        final String messageType,
        final Value payload,
        final List<HeaderField> headers)
    {
        Node node;

        if (payload == null) {
            node = NULL_NODE;
        } else {
            node = new PayloadNode(messageType, payload);
        }
        queue.add(node);
    }

    @Override
    protected void implShutdownInput() {
        super.implShutdownInput();
        queue.add(NULL_NODE);
    }

    public T get() throws CocaineException, IOException, InterruptedException {
        Node node;
        if (readable) {
            node = queue.poll(readTimeout, TimeUnit.MILLISECONDS);
        } else {
            node = queue.poll();
        }
        if (node == null && !readable) {
            node = NULL_NODE;
        }
        if (node == null) {
            throw new SocketTimeoutException();
        } else {
            return node.deserialize(deserializer);
        }
    }

    private interface Node {
        <T> T deserialize(CocainePayloadDeserializer<? extends T> deserializer)
            throws CocaineException, IOException;
    }

    private static class ErrorNode implements Node {
        private final CocaineException e;

        ErrorNode(final CocaineException e) {
            this.e = e;
        }

        @Override
        public <T> T deserialize(
            final CocainePayloadDeserializer<? extends T> deserializer)
            throws CocaineException
        {
            e.addSuppressed(new Exception());
            throw e;
        }
    }

    private static class PayloadNode implements Node {
        private final String messageType;
        private final Value payload;

        PayloadNode(final String messageType, final Value payload) {
            this.messageType = messageType;
            this.payload = payload;
        }

        @Override
        public <T> T deserialize(
            final CocainePayloadDeserializer<? extends T> deserializer)
            throws CocaineException, IOException
        {
            return deserializer.deserialize(messageType, payload);
        }
    }
}

