package ru.yandex.client.cocaine.logging;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.ErrorManager;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;

import cocaine.hpack.HeaderField;
import org.joda.time.DateTimeZone;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.io.StringBuilderWriter;

public class CocaineLoggingHandler extends Handler {
    private static final Formatter MESSAGE_FORMATTER = new Formatter() {
        @Override
        public String format(final LogRecord record) {
            String message = record.getMessage();
            Throwable thrown = record.getThrown();
            if (thrown == null) {
                return message;
            } else {
                StringBuilder sb = new StringBuilder(message);
                sb.append(':');
                sb.append(' ');
                thrown.printStackTrace(new StringBuilderWriter(sb));
                return new String(sb);
            }
        }
    };
    private static final DateTimeFormatter DATE_FORMATTER =
        DateTimeFormat.forPattern("YYYY-MM-dd'T'HH:mm:ss.SSSZ")
            .withZone(DateTimeZone.forID("Europe/Moscow"));

    private final CocaineLoggingService logging;
    private final List<List<String>> commonFlatAttrs;
    private final List<HeaderField> headers;

    public CocaineLoggingHandler(
        final CocaineLoggingService logging,
        final List<List<String>> commonFlatAttrs,
        final List<HeaderField> headers)
    {
        this.logging = logging;
        this.commonFlatAttrs = commonFlatAttrs;
        this.headers = headers;
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() {
    }

    @Override
    public Formatter getFormatter() {
        Formatter formatter = super.getFormatter();
        if (formatter == null) {
            formatter = MESSAGE_FORMATTER;
        }
        return formatter;
    }

    private static int convertLevel(final int level) {
        int result;
        if (level < Level.INFO.intValue()) {
            result = CocaineLoggingService.DEBUG;
        } else if (level < Level.WARNING.intValue()) {
            result = CocaineLoggingService.INFO;
        } else if (level < Level.SEVERE.intValue()) {
            result = CocaineLoggingService.WARNING;
        } else {
            result = CocaineLoggingService.ERROR;
        }
        return result;
    }

    @Override
    public void publish(final LogRecord record) {
        if (isLoggable(record)) {
            try {
                List<List<String>> flatAttrs =
                    new ArrayList<>(commonFlatAttrs.size() + 2);
                flatAttrs.addAll(commonFlatAttrs);
                flatAttrs.add(
                    Arrays.asList(
                        "server_timestamp",
                        DATE_FORMATTER.print(record.getMillis())));
                flatAttrs.add(logging.uuid());
                logging.append(
                    convertLevel(record.getLevel().intValue()),
                    getFormatter().format(record),
                    flatAttrs,
                    headers);
            } catch (CocaineException e) {
                reportError(null, e, ErrorManager.FORMAT_FAILURE);
            } catch (IOException e) {
                reportError(null, e, ErrorManager.WRITE_FAILURE);
            }
        }
    }
}

