package ru.yandex.client.cocaine.logging;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import cocaine.hpack.HeaderField;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineProtocolException;
import ru.yandex.client.cocaine.CocaineService;
import ru.yandex.client.cocaine.CocaineServiceFactory;
import ru.yandex.client.cocaine.CocaineSessionFactory;
import ru.yandex.client.cocaine.IntegerCocainePayloadDeserializer;
import ru.yandex.client.cocaine.PipedCocaineSession;
import ru.yandex.client.cocaine.PipedCocaineSessionFactory;
import ru.yandex.client.cocaine.VoidCocaineSessionFactory;
import ru.yandex.client.cocaine.protocol.CocaineServiceInfo;
import ru.yandex.function.GenericAutoCloseable;

public class CocaineLoggingService
    implements GenericAutoCloseable<IOException>
{
    public static final int DEBUG = 0;
    public static final int INFO = 1;
    public static final int WARNING = 2;
    public static final int ERROR = 3;

    private final int emitMethodId;
    private final int verbosityMethodId;
    private final CocaineService service;
    private final CocaineSessionFactory<PipedCocaineSession<Integer>>
        verbositySessionFactory;
    private final String appName;
    private final List<String> uuid;

    public CocaineLoggingService(
        final CocaineServiceInfo serviceInfo,
        final CocaineServiceFactory cocaineServiceFactory,
        final CocaineLoggingConfig config)
        throws IOException, CocaineProtocolException
    {
        // TODO: use only boxed values or use primitive to object map
        emitMethodId = serviceInfo.methodId("emit");
        verbosityMethodId = serviceInfo.methodId("verbosity");
        service = cocaineServiceFactory.createService(serviceInfo);
        verbositySessionFactory = new PipedCocaineSessionFactory<>(
            config.readTimeout(),
            IntegerCocainePayloadDeserializer.INSTANCE);
        appName = config.appName();
        uuid = new ArrayList<>(2);
        uuid.add("uuid");
        uuid.add(config.uuid());
    }

    public List<String> uuid() {
        return uuid;
    }

    public void start() {
        service.start();
    }

    @Override
    public void close() throws IOException {
        service.close();
    }

    public int verbosity(final List<HeaderField> headers)
        throws CocaineException, IOException, InterruptedException
    {
        try (PipedCocaineSession<Integer> session = service.createSession(
                verbositySessionFactory,
                verbosityMethodId,
                Collections.emptyList(),
                headers))
        {
            return session.get();
        }
    }

    private void append(
        final List<Object> args,
        final List<HeaderField> headers)
        throws CocaineException, IOException
    {
        service
            .createSession(
                VoidCocaineSessionFactory.INSTANCE,
                emitMethodId,
                args,
                headers)
            .close();
    }

    public void append(
        final int level,
        final String message,
        final List<HeaderField> headers)
        throws CocaineException, IOException
    {
        append(Arrays.asList(level, appName, message), headers);
    }

    // CSOFF: ParameterNumber
    public void append(
        final int level,
        final String message,
        final Map<String, ?> attrs,
        final List<HeaderField> headers)
        throws CocaineException, IOException
    {
        List<List<String>> flatAttrs = new ArrayList<>(attrs.size());
        for (Map.Entry<String, ?> entry: attrs.entrySet()) {
            flatAttrs.add(
                Arrays.asList(
                    entry.getKey(),
                    Objects.toString(entry.getValue())));
        }
        append(level, message, flatAttrs, headers);
    }

    public void append(
        final int level,
        final String message,
        final List<List<String>> flatAttrs,
        final List<HeaderField> headers)
        throws CocaineException, IOException
    {
        if (flatAttrs.isEmpty()) {
            append(level, message, headers);
        } else {
            append(Arrays.asList(level, appName, message, flatAttrs), headers);
        }
    }
    // CSON: ParameterNumber
}

