package ru.yandex.client.cocaine.protocol;

import java.io.IOException;
import java.net.SocketAddress;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

import cocaine.msgpack.SocketAddressTemplate;
import org.msgpack.template.Templates;
import org.msgpack.type.Value;
import org.msgpack.unpacker.Converter;
import org.msgpack.unpacker.Unpacker;

import ru.yandex.client.cocaine.CocainePayloadDeserializer;
import ru.yandex.client.cocaine.CocaineProtocolException;
import ru.yandex.client.cocaine.protocol.fsm.BasicFsmState;
import ru.yandex.client.cocaine.protocol.fsm.BasicFsmTransition;
import ru.yandex.client.cocaine.protocol.fsm.FsmState;
import ru.yandex.function.CollectionConsumer;
import ru.yandex.function.NullConsumer;

public class CocaineServiceInfoDeserializer
    implements CocainePayloadDeserializer<CocaineServiceInfo>
{
    private final CocaineProtocolRegistry cocaineProtocolRegistry;

    public CocaineServiceInfoDeserializer(
        final CocaineProtocolRegistry cocaineProtocolRegistry)
    {
        this.cocaineProtocolRegistry = cocaineProtocolRegistry;
    }

    @Override
    public CocaineServiceInfo deserialize(
        final String messageType,
        final Value payload)
        throws IOException, CocaineProtocolException
    {
        Unpacker unpacker = new Converter(payload);
        // ignore array length
        unpacker.readArrayBegin();

        List<SocketAddress> endpoints = unpacker.read(
            Templates.tList(SocketAddressTemplate.getInstance()));
        if (endpoints.isEmpty()) {
            throw new CocaineProtocolException(
                "No endpoints found for service");
        }

        // ignore version
        unpacker.readInt();

        int length = unpacker.readMapBegin();
        Map<String, Integer> methodNameToId = new HashMap<>(length << 1);
        Map<Integer, CocaineMethodApi> methods = new HashMap<>(length << 1);
        for (int i = 0; i < length; ++i) {
            Integer id = unpacker.readInt();
            // ignore array length
            unpacker.readArrayBegin();
            methodNameToId.put(unpacker.readString(), id);
            FsmState transmitState = deserializeState(
                unpacker,
                NullConsumer.INSTANCE);
            Set<String> allMessages = new HashSet<>();
            FsmState receiveState = deserializeState(
                unpacker,
                new CollectionConsumer<>(allMessages));
            unpacker.readArrayEnd();
            methods.put(
                id,
                new CocaineMethodApi(
                    cocaineProtocolRegistry.findProtocol(allMessages),
                    transmitState,
                    receiveState));
        }
        unpacker.readMapEnd();

        unpacker.readArrayEnd();

        return new CocaineServiceInfo(endpoints, methodNameToId, methods);
    }

    private static FsmState deserializeState(
        final Unpacker unpacker,
        final Consumer<? super String> messagesConsumer)
        throws IOException
    {
        int length = unpacker.readMapBegin();
        BasicFsmState state = new BasicFsmState(length);
        for (int i = 0; i < length; ++i) {
            int messageId = unpacker.readInt();
            // ignore array length
            unpacker.readArrayBegin();
            String messageName = unpacker.readString();
            messagesConsumer.accept(messageName);
            FsmState nextState;
            if (unpacker.trySkipNil()) {
                nextState = state;
            } else {
                nextState = deserializeState(unpacker, messagesConsumer);
            }
            state.addTransition(
                new BasicFsmTransition(messageId, messageName, nextState));
            unpacker.readArrayEnd();
        }
        unpacker.readMapEnd();
        return state;
    }
}

