package ru.yandex.client.cocaine.protocol;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.msgpack.type.ArrayValue;
import org.msgpack.type.Value;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineProtocolException;
import ru.yandex.client.cocaine.CocaineServiceException;

public enum DefaultCocaineProtocolRegistry implements CocaineProtocolRegistry {
    INSTANCE;

    // Common message type
    public static final String ERROR = "error";
    // Primitive protocol message
    public static final String VALUE = "value";
    // Streaming protocol messages
    public static final String WRITE = "write";
    public static final String CLOSE = "close";
    // Streaming read protocol messages
    public static final String CHUNK = "chunk";
    public static final String END = "end";
    public static final String META = "meta";
    // Service session messages
    public static final String HANDSHAKE = "handshake";
    public static final String HEARTBEAT = "heartbeat";
    public static final String TERMINATE = "terminate";

    public enum Protocols implements CocaineProtocol {
        IDENTITY {
            @Override
            public Value handle(final String messageType, final Value payload)
                throws CocaineException
            {
                return payload;
            }
        },
        PRIMITIVE {
            @Override
            public Value handle(final String messageType, final Value payload)
                throws CocaineException
            {
                if (VALUE.equals(messageType)) {
                    return unpackArray(payload);
                }
                throw error(messageType, payload);
            }
        },
        STREAMING {
            @Override
            public Value handle(final String messageType, final Value payload)
                throws CocaineException
            {
                if (WRITE.equals(messageType)) {
                    return unpackArray(payload);
                }
                if (CLOSE.equals(messageType)) {
                    return null;
                }
                throw error(messageType, payload);
            }
        },
        STREAMING_READ {
            @Override
            public Value handle(final String messageType, final Value payload)
                throws CocaineException
            {
                if (CHUNK.equals(messageType) || META.equals(messageType)) {
                    return unpackArray(payload);
                }
                if (END.equals(messageType)) {
                    return null;
                }
                throw error(messageType, payload);
            }
        };

        private static Value unpackArray(final Value payload)
            throws CocaineProtocolException
        {
            if (!payload.isArrayValue()) {
                throw new CocaineProtocolException(payload);
            }
            ArrayValue values = payload.asArrayValue();
            if (values.size() == 1) {
                return values.get(0);
            } else {
                return values;
            }
        }

        private static CocaineException error(
            final String messageType,
            final Value payload)
        {
            if (ERROR.equals(messageType) && payload.isArrayValue()) {
                ArrayValue values = payload.asArrayValue();
                if (values.size() == 2) {
                    Value codes = values.get(0);
                    Value message = values.get(1);
                    if (codes.isArrayValue() && message.isRawValue()) {
                        ArrayValue codesArray = codes.asArrayValue();
                        Value category = codesArray.get(0);
                        Value code = codesArray.get(1);
                        if (category.isIntegerValue()
                            && code.isIntegerValue())
                        {
                            return new CocaineServiceException(
                                category.asIntegerValue().getInt(),
                                code.asIntegerValue().getInt(),
                                message.toString());
                        }
                    }
                }
            }
            return new CocaineProtocolException(messageType, payload);
        }
    }

    @SuppressWarnings("ImmutableEnumChecker")
    private final Map<Set<String>, CocaineProtocol> protocols =
        new HashMap<>();

    DefaultCocaineProtocolRegistry() {
        protocols.put(
            new HashSet<>(Arrays.asList(ERROR, VALUE)),
            Protocols.PRIMITIVE);
        protocols.put(
            new HashSet<>(Arrays.asList(ERROR, WRITE, CLOSE)),
            Protocols.STREAMING);
        protocols.put(
            new HashSet<>(Arrays.asList(ERROR, CHUNK, END, META)),
            Protocols.STREAMING_READ);
    }

    @Override
    public CocaineProtocol findProtocol(final Set<String> methods) {
        return protocols.getOrDefault(methods, Protocols.IDENTITY);
    }
}

