package ru.yandex.client.cocaine.unistorage;

import java.io.IOException;
import java.io.InputStream;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.function.EmptyProcessable;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.function.Processable;

public class UnistorageInputStream extends InputStream {
    private static final int BYTE_MASK = 0xff;

    private final byte[] buf = new byte[1];
    private final UnistorageSession session;
    private Processable<byte[]> data = EmptyProcessable.instance();
    private boolean closed = false;

    public UnistorageInputStream(final UnistorageSession session)
        throws IOException
    {
        try (GenericAutoCloseableHolder<IOException, UnistorageSession>
                holder = new GenericAutoCloseableHolder<>(session))
        {
            if (session.prefetch()) {
                session.requestChunk();
            }
            this.session = holder.release();
        }
    }

    @Override
    public int available() {
        if (data == null) {
            return 0;
        } else {
            return data.length();
        }
    }

    @Override
    public void close() throws IOException {
        if (!closed) {
            closed = true;
            session.close();
        }
    }

    @Override
    public int read() throws IOException {
        int read = read(buf, 0, 1);
        if (read == -1) {
            return -1;
        }
        return buf[0] & BYTE_MASK;
    }

    @Override
    public int read(final byte[] buf) throws IOException {
        return read(buf, 0, buf.length);
    }

    @Override
    public int read(final byte[] buf, final int off, final int len)
        throws IOException
    {
        if (closed) {
            throw new IOException("Stream already closed");
        }
        int result;
        if (data == null) {
            result = -1;
        } else if (data.isEmpty()) {
            if (!session.prefetch()) {
                session.requestChunk();
            }
            try {
                UnistorageResponse response = session.get();
                if (response == null) {
                    data = null;
                } else {
                    data = response.asData();
                }
            } catch (CocaineException | InterruptedException e) {
                throw new IOException(e);
            }
            result = read(buf, off, len);
        } else {
            result = data.transferTo(buf, off, len);
            if (session.prefetch() && data.isEmpty()) {
                session.requestChunk();
            }
        }
        return result;
    }
}

