package ru.yandex.client.cocaine.worker.http;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import cocaine.hpack.HeaderField;
import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;

import ru.yandex.client.cocaine.CocaineServiceException;
import ru.yandex.client.cocaine.logging.CocaineLoggingHandler;
import ru.yandex.client.cocaine.logging.CocaineLoggingService;
import ru.yandex.client.cocaine.worker.CocaineWorkerSession;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.logger.BackendAccessLoggerConfigDefaults;
import ru.yandex.logger.IdGenerator;
import ru.yandex.parser.uri.ScanningCgiParams;
import ru.yandex.util.string.HexStrings;

public abstract class AbstractCocaineHttpRequestHandler
    implements CocaineHttpRequestHandler
{
    private final IdGenerator idGenerator = new IdGenerator();
    private final CocaineLoggingService logging;
    private final long minRequestTime;

    protected AbstractCocaineHttpRequestHandler(
        final CocaineLoggingService logging,
        final long minRequestTime)
    {
        this.logging = logging;
        this.minRequestTime = minRequestTime;
    }

    private static void appendBytes(final StringBuilder sb, final byte[] buf) {
        for (int i = 0; i < buf.length; ++i) {
            if (buf[i] < ' ') {
                sb.append('0');
                sb.append('x');
                HexStrings.LOWER.toStringBuilder(sb, buf);
                return;
            }
        }
        sb.append('"');
        for (byte b: buf) {
            sb.append((char) b);
        }
        sb.append('"');
    }

    private static void appendHeader(
        final StringBuilder sb,
        final HeaderField header)
    {
        int nameStart = sb.length() + 1;
        appendBytes(sb, header.name);
        String name = new String(sb).substring(nameStart, sb.length() - 1);
        if (YandexHeaders.DEFAULT_HIDDEN_HEADERS.contains(name)) {
            sb.append(": ...");
        } else {
            sb.append(':');
            sb.append(' ');
            appendBytes(sb, header.value);
        }
    }

    // CSOFF: ParameterNumber
    protected abstract void handle(
        final HttpRequest request,
        final List<HeaderField> headers,
        final CocaineWorkerSession session,
        final Logger logger)
        throws HttpException;
    // CSON: ParameterNumber

    @Override
    public void handle(
        final HttpRequest request,
        final List<HeaderField> headers,
        final CocaineWorkerSession session)
    {
        Logger logger = Logger.getAnonymousLogger();
        List<List<String>> loggerAttrs = new ArrayList<>(2);
        loggerAttrs.add(Arrays.asList("session_id", idGenerator.next()));
        Header proxySessionId = request.getFirstHeader(
            BackendAccessLoggerConfigDefaults.X_PROXY_SESSION_ID);
        if (proxySessionId != null) {
            loggerAttrs.add(
                Arrays.asList("proxy_session_id", proxySessionId.getValue()));
        }
        loggerAttrs.add(
            Arrays.asList("thread_name", Thread.currentThread().getName()));
        logger.addHandler(
            new CocaineLoggingHandler(
                logging,
                loggerAttrs,
                session.headers()));
        logger.setLevel(Level.ALL);
        logger.setUseParentHandlers(false);
        long start = System.currentTimeMillis();
        try (CocaineWorkerSession workerSession = session) {
            logger.info("Processing request: " + request);
            StringBuilder sb = new StringBuilder("Cocaine session age: ");
            sb.append(System.currentTimeMillis() - session.bornDate());
            sb.append(" ms, session headers: ");
            for (HeaderField header: session.headers()) {
                appendHeader(sb, header);
                sb.append(',');
                sb.append(' ');
            }
            sb.append("request headers: ");
            for (HeaderField header: headers) {
                appendHeader(sb, header);
                sb.append(',');
                sb.append(' ');
            }
            sb.setLength(sb.length() - 2);
            logger.info(new String(sb));
            try {
                Header referer =
                    request.getFirstHeader(HttpHeaders.REFERER);
                if (referer != null) {
                    Long deadline =
                        new ScanningCgiParams(referer.getValue())
                            .getLast("deadline", null, Long::valueOf);
                    if (deadline != null) {
                        long expectedCompletionTime =
                            System.currentTimeMillis() + minRequestTime;
                        if (expectedCompletionTime > deadline) {
                            throw new ServerException(
                                YandexHttpStatus.SC_BUSY,
                                "No chance to process request before "
                                + deadline);
                        }
                    }
                }
                try {
                    handle(request, headers, workerSession, logger);
                } catch (RuntimeException e) {
                    throw new ServiceUnavailableException(e);
                }
            } catch (ServerException e) {
                logger.log(Level.WARNING, "Server exception occured", e);
                new CocaineDummyHttpRequestHandler(
                    e.statusCode(),
                    e.toStackTrace())
                    .handle(request, headers, workerSession);
            } catch (HttpException e) {
                logger.log(Level.WARNING, "HTTP exception occured", e);
                StringBuilderWriter sbw = new StringBuilderWriter();
                e.printStackTrace(sbw);
                new CocaineDummyHttpRequestHandler(
                    YandexHttpStatus.SC_SERVICE_UNAVAILABLE,
                    sbw.toString())
                    .handle(request, headers, workerSession);
            }
        }
        logger.info(
            "Request " + request + " processing time: "
            + (System.currentTimeMillis() - start) + " ms");
    }

    public static void unwrapCocaineServiceException(final IOException e)
        throws HttpException
    {
        unwrapCocaineServiceException(e, e.getCause());
    }

    private static void unwrapCocaineServiceException(
        final IOException e,
        final Throwable cause)
        throws HttpException
    {
        if (cause != null) {
            if (cause instanceof CocaineServiceException) {
                HttpException ex = CocaineServiceExceptionConverter.INSTANCE
                    .apply((CocaineServiceException) cause);
                ex.addSuppressed(e);
                throw ex;
            } else {
                unwrapCocaineServiceException(e, cause.getCause());
            }
        }
    }
}

