package ru.yandex.client.cocaine.worker.http;

import java.io.IOException;
import java.util.List;

import cocaine.hpack.HeaderField;
import org.apache.http.Header;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.MethodNotSupportedException;
import org.apache.http.ParseException;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.impl.DefaultHttpRequestFactory;
import org.apache.http.message.BasicLineParser;
import org.apache.http.message.BasicRequestLine;
import org.msgpack.MessagePack;
import org.msgpack.type.RawValue;
import org.msgpack.unpacker.Unpacker;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineProtocolException;
import ru.yandex.client.cocaine.worker.CocaineEventConsumer;
import ru.yandex.client.cocaine.worker.CocaineEventHandler;
import ru.yandex.client.cocaine.worker.CocaineInputStreamConsumer;
import ru.yandex.client.cocaine.worker.CocaineWorkerSession;
import ru.yandex.http.util.YandexHeaders;

public class CocaineHttpEventHandler implements CocaineEventHandler {
    private static final MessagePack MESSAGE_PACK = new MessagePack();

    private final CocaineHttpRequestHandler handler;
    private final long readTimeout;

    public CocaineHttpEventHandler(
        final CocaineHttpRequestHandler handler,
        final long readTimeout)
    {
        this.handler = handler;
        this.readTimeout = readTimeout;
    }

    @Override
    public CocaineEventConsumer handle(
        final RawValue payload,
        final List<HeaderField> headers,
        final CocaineWorkerSession session)
        throws CocaineException
    {
        Unpacker unpacker =
            MESSAGE_PACK.createBufferUnpacker(payload.getByteArray());
        try {
            unpacker.readArrayBegin();
            String method = unpacker.readString();
            String uri = unpacker.readString();
            String version = "HTTP/" + unpacker.readString();
            HttpRequest request =
                DefaultHttpRequestFactory.INSTANCE.newHttpRequest(
                    new BasicRequestLine(
                        method,
                        uri,
                        BasicLineParser.parseProtocolVersion(version, null)));

            int headersCount = unpacker.readArrayBegin();
            for (int i = 0; i < headersCount; ++i) {
                unpacker.readArrayBegin();
                String name = unpacker.readString();
                String value = unpacker.readString();
                unpacker.readArrayEnd();
                if (YandexHeaders.DEFAULT_HIDDEN_HEADERS.contains(name)) {
                    value = "...";
                }
                request.addHeader(name, value);
            }
            unpacker.readArrayEnd();
            CocaineEventConsumer eventConsumer;
            if (request instanceof HttpEntityEnclosingRequest) {
                CocaineInputStreamConsumer consumer =
                    new CocaineInputStreamConsumer(readTimeout, unpacker);
                eventConsumer = consumer;
                long contentLength = -1L;
                Header header =
                    request.getFirstHeader(HttpHeaders.CONTENT_LENGTH);
                if (header != null) {
                    contentLength = Long.parseLong(header.getValue());
                }
                InputStreamEntity entity =
                    new InputStreamEntity(consumer, contentLength);
                header = request.getFirstHeader(HttpHeaders.CONTENT_TYPE);
                if (header != null) {
                    entity.setContentType(header);
                }
                header = request.getFirstHeader(HttpHeaders.CONTENT_ENCODING);
                if (header != null) {
                    entity.setContentEncoding(header);
                }
                ((HttpEntityEnclosingRequest) request).setEntity(entity);
            } else {
                eventConsumer = CocaineEventConsumer.EMPTY;
                unpacker.readByteArray();
            }
            unpacker.readArrayEnd();
            handler.handle(request, headers, session);
            return eventConsumer;
        } catch (IOException
            | ParseException
            | NumberFormatException
            | MethodNotSupportedException e)
        {
            throw new CocaineProtocolException(e);
        }
    }
}

