package ru.yandex.client.cocaine.worker.http;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.apache.http.Header;
import org.msgpack.MessagePack;
import org.msgpack.packer.BufferPacker;

import ru.yandex.client.cocaine.worker.CocaineWorkerService;
import ru.yandex.client.cocaine.worker.CocaineWorkerSession;
import ru.yandex.http.util.YandexHeaders;

public class CocaineHttpOutputStream extends OutputStream {
    private static final MessagePack MESSAGE_PACK = new MessagePack();

    private final byte[] buf = new byte[1];
    private final CocaineWorkerSession session;
    private boolean opened = true;

    // CSOFF: ParameterNumber
    public CocaineHttpOutputStream(
        final CocaineWorkerSession session,
        final int statusCode,
        final List<Header> headers)
        throws IOException
    {
        this.session = session;
        BufferPacker packer = MESSAGE_PACK.createBufferPacker();
        packer.writeArrayBegin(2);
        packer.write(statusCode);
        packer.writeArrayBegin(headers.size() + 1);
        writeHeader(packer, "Built-Date", YandexHeaders.BUILT_DATE);
        for (Header header: headers) {
            writeHeader(packer, header.getName(), header.getValue());
        }
        packer.writeArrayEnd();
        packer.writeArrayEnd();
        session.sendMessage(
            CocaineWorkerService.WRITE_MESSAGE_ID,
            Collections.singletonList(packer.toByteArray()));
    }
    // CSON: ParameterNumber

    private static void writeHeader(
        final BufferPacker packer,
        final String name,
        final String value)
        throws IOException
    {
        packer.writeArrayBegin(2);
        packer.write(name);
        packer.write(value);
        packer.writeArrayEnd();
    }

    // Something gone wrong
    public void sendError(
        final int category,
        final int code,
        final String message)
        throws IOException
    {
        if (opened) {
            opened = false;
            try (CocaineWorkerSession session = this.session) {
                session.sendMessage(
                    CocaineWorkerService.ERROR_MESSAGE_ID,
                    Arrays.asList(Arrays.asList(category, code), message));
            }
        }
    }

    // Processing done, no more data expected to be written
    public void sendEof() throws IOException {
        if (opened) {
            opened = false;
            try (CocaineWorkerSession session = this.session) {
                session.sendMessage(
                    CocaineWorkerService.CLOSE_MESSAGE_ID,
                    Collections.emptyList());
            }
        }
    }

    @Override
    public void close() throws IOException {
        if (opened) {
            opened = false;
            session.close();
        }
    }

    @Override
    public void write(final int b) throws IOException {
        buf[0] = (byte) b;
        write(buf, 0, 1);
    }

    @Override
    public void write(final byte[] buf) throws IOException {
        write(buf, 0, buf.length);
    }

    @Override
    public void write(final byte[] buf, final int off, final int len)
        throws IOException
    {
        session.sendMessage(
            CocaineWorkerService.WRITE_MESSAGE_ID,
            Collections.singletonList(
                Arrays.copyOfRange(buf, off, off + len)));
    }
}

