package ru.yandex.client.cocaine.worker.http.unistorage;

import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import ru.yandex.client.cocaine.CocaineClientFactoryFactory;
import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineServiceFactory;
import ru.yandex.client.cocaine.IOErrorHandler;
import ru.yandex.client.cocaine.logging.CocaineLoggingConfig;
import ru.yandex.client.cocaine.logging.CocaineLoggingService;
import ru.yandex.client.cocaine.protocol.DefaultCocaineProtocolRegistry;
import ru.yandex.client.cocaine.protocol.LocatorService;
import ru.yandex.client.cocaine.protocol.LocatorServiceConfig;
import ru.yandex.client.cocaine.unistorage.UnistorageService;
import ru.yandex.client.cocaine.worker.CocaineMapEventHandlerRegistry;
import ru.yandex.client.cocaine.worker.CocaineWorkerConfig;
import ru.yandex.client.cocaine.worker.CocaineWorkerService;
import ru.yandex.client.cocaine.worker.CocaineWorkerSystemSession;
import ru.yandex.collection.BlockingBlockingQueue;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.function.GenericNonThrowingCloseableAdapter;

// TODO: should extend CocaineHttpService
public class UnistorageHttpService
    implements GenericAutoCloseable<IOException>,
        IOErrorHandler,
        Thread.UncaughtExceptionHandler
{
    protected final AtomicBoolean closed = new AtomicBoolean();
    protected final CocaineMapEventHandlerRegistry handlerRegistry =
        new CocaineMapEventHandlerRegistry();
    protected final Runnable terminateCallback = () -> terminate();
    protected final ImmutableUnistorageHttpServiceConfig serviceConfig;
    protected final CocaineWorkerConfig workerConfig;
    protected final ThreadPoolExecutor executor;
    protected final TerminationThread terminator;
    protected final CocaineLoggingService logging;
    protected final UnistorageService unistorage;
    protected final CocaineWorkerService worker;
    protected final GenericAutoCloseableChain<IOException> chain;

    public UnistorageHttpService(
        final ImmutableUnistorageHttpServiceConfig serviceConfig,
        final CocaineWorkerConfig workerConfig,
        final String serviceName)
        throws CocaineException, IOException, InterruptedException
    {
        this.serviceConfig = serviceConfig;
        this.workerConfig = workerConfig;
        ThreadGroup threadGroup = Thread.currentThread().getThreadGroup();
        executor = new ThreadPoolExecutor(
            serviceConfig.workers(),
            serviceConfig.workers(),
            1,
            TimeUnit.MINUTES,
            new BlockingBlockingQueue<>(
                new LifoWaitBlockingQueue<>(
                    serviceConfig.requestsQueueSize())),
            new NamedThreadFactory(
                new ThreadFactoryConfig(serviceName + '-')
                    .group(threadGroup)
                    .daemon(true)
                    .uncaughtExceptionHandler(this)));
        terminator = new TerminationThread(
            threadGroup,
            serviceName,
            serviceConfig.gracefulShutdownTimeout());
        try (GenericAutoCloseableHolder<
                IOException,
                GenericAutoCloseableChain<IOException>> chain =
                    new GenericAutoCloseableHolder<>(
                        new GenericAutoCloseableChain<>()))
        {
            try (LocatorService locator =
                    new LocatorService(
                        new LocatorServiceConfig(
                            workerConfig.locatorEndpoint(),
                            serviceConfig.locatorConfig().readTimeout()),
                        DefaultCocaineProtocolRegistry.INSTANCE,
                        new CocaineServiceFactory(
                            serviceConfig.locatorConfig().connectTimeout(),
                            this,
                            new CocaineClientFactoryFactory(
                                new ThreadFactoryConfig(
                                    serviceName + "-LocatorService")
                                    .group(threadGroup)
                                    .daemon(true)
                                    .uncaughtExceptionHandler(this))
                                .create(
                                    serviceConfig.locatorConfig()
                                        .queueSize()))))
            {
                locator.start();

                logging = new CocaineLoggingService(
                    locator.resolve("logging"),
                    new CocaineServiceFactory(
                        serviceConfig.loggingConfig().connectTimeout(),
                        this,
                        new CocaineClientFactoryFactory(
                            new ThreadFactoryConfig(serviceName + "-Logging")
                                .group(threadGroup)
                                .daemon(true)
                                .uncaughtExceptionHandler(this))
                            .create(
                                serviceConfig.loggingConfig().queueSize())),
                    new CocaineLoggingConfig(
                        serviceConfig.loggingConfig().readTimeout(),
                        workerConfig.appName(),
                        workerConfig.uuid()));
                chain.get().add(logging);

                unistorage = new UnistorageService(
                    locator.resolve("unistorage"),
                    new CocaineServiceFactory(
                        serviceConfig.unistorageConfig().connectTimeout(),
                        this,
                        new CocaineClientFactoryFactory(
                            new ThreadFactoryConfig(
                                serviceName + "-UnistorageService")
                                .group(threadGroup)
                                .daemon(true)
                                .uncaughtExceptionHandler(this))
                            .create(
                                serviceConfig.unistorageConfig().queueSize())),
                    serviceConfig.unistorageConfig());
                chain.get().add(unistorage);

                worker = new CocaineWorkerService(
                    this,
                    CocaineWorkerService.createWorkerServiceContext(
                        workerConfig.workerEndpoint(),
                        new CocaineClientFactoryFactory(
                            new ThreadFactoryConfig(serviceName)
                                .group(threadGroup)
                                .daemon(false)
                                .uncaughtExceptionHandler(this))
                            .create(serviceConfig.queueSize())),
                    handlerRegistry);
                chain.get().add(worker);
            }
            this.chain = chain.release();
        }
    }

    public void start() throws CocaineException, IOException {
        executor.prestartAllCoreThreads();
        terminator.start();
        logging.start();
        unistorage.start();
        worker.start();
        CocaineWorkerSystemSession systemSession =
            worker.createSession(
                new CocaineWorkerSystemSession.Factory(
                    terminateCallback,
                    executor,
                    serviceConfig),
                CocaineWorkerService.HANDSHAKE_METHOD_ID,
                Collections.singletonList(workerConfig.uuid()));
        chain.add(new GenericNonThrowingCloseableAdapter<>(systemSession));
        systemSession.scheduleHeartbeat(0L);
    }

    @Override
    public void close() throws IOException {
        if (closed.compareAndSet(false, true)) {
            terminator.terminated = true;
            executor.shutdown();
            chain.close();
        }
    }

    @Override
    public void readFailed(final IOException e) {
        uncaughtException(Thread.currentThread(), e);
    }

    @Override
    public void writeFailed(final IOException e) {
        uncaughtException(Thread.currentThread(), e);
    }

    @Override
    public void uncaughtException(
        final Thread thread,
        final Throwable t)
    {
        if (!terminator.terminated) {
            terminator.terminated = true;
            dumpThreads();
            new Exception(
                "Shutdown caused by exception in thread " + thread.getName(),
                t)
                .printStackTrace();
            closeQuietly();
        }
    }

    private void terminate() {
        if (!terminator.terminated) {
            terminator.terminated = true;
            dumpThreads();
            new Exception(
                "Server shutdown requested in thread "
                + Thread.currentThread().getName())
                .printStackTrace();
            closeQuietly();
        }
    }

    private void dumpThreads() {
        System.err.println("Threads status");
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<Thread, StackTraceElement[]> entry
            : Thread.getAllStackTraces().entrySet())
        {
            Thread thread = entry.getKey();
            sb.setLength(0);
            sb.append("Thread ");
            sb.append(thread.getName());
            sb.append('(');
            sb.append(thread.getThreadGroup().getName());
            sb.append(',');
            sb.append(thread.getState());
            sb.append(',');
            sb.append(thread.isDaemon());
            sb.append(')');
            sb.append(':');
            for (StackTraceElement frame: entry.getValue()) {
                sb.append(' ');
                sb.append(frame);
            }
            System.err.println(sb.toString());
        }
        System.err.println();
    }

    private void closeQuietly() {
        try {
            close();
        } catch (IOException e) {
            e.printStackTrace();
            Runtime.getRuntime().halt(1);
        }
    }

    private static class TerminationThread extends Thread {
        private final long gracefulShutdownTimeout;
        private volatile boolean terminated = false;

        TerminationThread(
            final ThreadGroup threadGroup,
            final String serviceName,
            final long gracefulShutdownTimeout)
        {
            super(threadGroup, serviceName + "-Terminator");
            this.gracefulShutdownTimeout = gracefulShutdownTimeout;
        }

        @Override
        public void run() {
            Runtime runtime = Runtime.getRuntime();
            try {
                while (!terminated) {
                    Thread.sleep(gracefulShutdownTimeout);
                }
                Thread.sleep(gracefulShutdownTimeout);
            } catch (InterruptedException e) {
                // Impossible situation
            }
            runtime.halt(2);
        }
    }
}

