package ru.yandex.client.cocaine.worker;

import java.io.IOException;
import java.io.InputStream;
import java.net.SocketTimeoutException;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

import cocaine.hpack.HeaderField;
import org.msgpack.type.RawValue;
import org.msgpack.unpacker.Unpacker;

import ru.yandex.client.cocaine.CocaineException;

public class CocaineInputStreamConsumer
    extends InputStream
    implements CocaineEventConsumer
{
    private static final Node EOF_NODE = new Node() {
        @Override
        public boolean isEmpty() {
            return false;
        }

        @Override
        public int read(final byte[] buf, final int off, final int len) {
            return -1;
        }
    };

    private static final int BYTE_MASK = 0xff;

    private final byte[] buf = new byte[1];
    private final BlockingQueue<Node> queue = new LinkedBlockingQueue<>();
    private final long readTimeout;
    private Node currentNode;

    public CocaineInputStreamConsumer(
        final long readTimeout,
        final Unpacker unpacker)
        throws IOException
    {
        this.readTimeout = readTimeout;
        currentNode = new DataNode(unpacker);
    }

    @Override
    public int read() throws IOException {
        int read = read(buf, 0, 1);
        if (read == -1) {
            return -1;
        }
        return buf[0] & BYTE_MASK;
    }

    @Override
    public int read(final byte[] buf) throws IOException {
        return read(buf, 0, buf.length);
    }

    @Override
    public int read(final byte[] buf, final int off, final int len)
        throws IOException
    {
        while (currentNode.isEmpty()) {
            Node node;
            try {
                node = queue.poll(readTimeout, TimeUnit.MILLISECONDS);
            } catch (InterruptedException e) {
                throw new SocketTimeoutException();
            }
            if (node == null) {
                throw new SocketTimeoutException();
            } else {
                currentNode = node;
            }
        }
        return currentNode.read(buf, off, len);
    }

    @Override
    public void accept(
        final RawValue payload,
        final List<HeaderField> headers)
    {
        queue.add(new DataNode(payload));
    }

    @Override
    public void eof() {
        queue.add(EOF_NODE);
    }

    @Override
    public void failed(final CocaineException e) {
        queue.add(new ErrorNode(e));
    }

    private interface Node {
        boolean isEmpty();

        int read(byte[] buf, int off, int len) throws IOException;
    }

    private static class ErrorNode implements Node {
        private final CocaineException e;

        ErrorNode(final CocaineException e) {
            this.e = e;
        }

        @Override
        public boolean isEmpty() {
            return false;
        }

        @Override
        public int read(final byte[] buf, final int off, final int len)
            throws IOException
        {
            throw new IOException(e);
        }
    }

    private static class DataNode implements Node {
        private final byte[] buf;
        private int pos = 0;

        DataNode(final Unpacker unpacker) throws IOException {
            buf = unpacker.readByteArray();
        }

        DataNode(final RawValue payload) {
            buf = payload.getByteArray();
        }

        @Override
        public boolean isEmpty() {
            return pos == buf.length;
        }

        @Override
        public int read(final byte[] buf, final int off, final int len) {
            int transferCount = Math.min(len, buf.length - pos);
            System.arraycopy(this.buf, pos, buf, off, transferCount);
            pos += transferCount;
            return transferCount;
        }
    }
}

