package ru.yandex.client.cocaine.worker;

import java.io.IOException;
import java.util.Collections;

import cocaine.message.Message;

import ru.yandex.client.cocaine.CocaineClientFactory;
import ru.yandex.client.cocaine.CocaineService;
import ru.yandex.client.cocaine.CocaineServiceContext;
import ru.yandex.client.cocaine.CocaineSession;
import ru.yandex.client.cocaine.CocaineSessionContext;
import ru.yandex.client.cocaine.IOErrorHandler;
import ru.yandex.client.cocaine.UniSocket;
import ru.yandex.client.cocaine.protocol.CocaineMethodApi;
import ru.yandex.client.cocaine.protocol.DefaultCocaineProtocolRegistry;
import ru.yandex.client.cocaine.protocol.fsm.BasicFsmState;
import ru.yandex.client.cocaine.protocol.fsm.BasicFsmTransition;
import ru.yandex.client.cocaine.protocol.fsm.FsmState;

public class CocaineWorkerService extends CocaineService {
    public static final int HANDSHAKE_METHOD_ID = 0;
    public static final int HEARTBEAT_MESSAGE_ID = 0;
    public static final int TERMINATE_MESSAGE_ID = 1;

    public static final int WRITE_MESSAGE_ID = 0;
    public static final int ERROR_MESSAGE_ID = 1;
    public static final int CLOSE_MESSAGE_ID = 2;

    private static final int INVOKE_METHOD_ID = 0;
    private static final CocaineMethodApi INVOKE_METHOD_API =
        createInvokeMethodApi();

    private final CocaineEventHandlerRegistry eventHandlerRegistry;

    public CocaineWorkerService(
        final IOErrorHandler ioErrorHandler,
        final CocaineServiceContext context,
        final CocaineEventHandlerRegistry eventHandlerRegistry)
    {
        super(ioErrorHandler, context);
        this.eventHandlerRegistry = eventHandlerRegistry;
    }

    public static CocaineServiceContext createWorkerServiceContext(
        final String endpoint,
        final CocaineClientFactory clientFactory)
        throws IOException
    {
        BasicFsmState transmitState = new BasicFsmState(1);
        transmitState.addTransition(
            new BasicFsmTransition(
                HEARTBEAT_MESSAGE_ID,
                DefaultCocaineProtocolRegistry.HEARTBEAT,
                transmitState));
        BasicFsmState receiveState = new BasicFsmState(2);
        receiveState.addTransition(
            new BasicFsmTransition(
                HEARTBEAT_MESSAGE_ID,
                DefaultCocaineProtocolRegistry.HEARTBEAT,
                receiveState));
        receiveState.addTransition(
            new BasicFsmTransition(
                TERMINATE_MESSAGE_ID,
                DefaultCocaineProtocolRegistry.TERMINATE,
                FsmState.FINAL));
        CloseableUnixDomainSocketClient socket =
            new CloseableUnixDomainSocketClient(endpoint);
        return new CocaineServiceContext(
            new UniSocket(
                socket,
                socket.getInputStream(),
                socket.getOutputStream()),
            Collections.singletonMap(
                0,
                new CocaineMethodApi(
                    DefaultCocaineProtocolRegistry.Protocols.IDENTITY,
                    transmitState,
                    receiveState)),
            clientFactory);
    }

    private static CocaineMethodApi createInvokeMethodApi() {
        BasicFsmState initialState = new BasicFsmState(2 + 1);
        initialState.addTransition(
            new BasicFsmTransition(
                WRITE_MESSAGE_ID,
                DefaultCocaineProtocolRegistry.WRITE,
                initialState));
        initialState.addTransition(
            new BasicFsmTransition(
                ERROR_MESSAGE_ID,
                DefaultCocaineProtocolRegistry.ERROR,
                FsmState.FINAL));
        initialState.addTransition(
            new BasicFsmTransition(
                CLOSE_MESSAGE_ID,
                DefaultCocaineProtocolRegistry.CLOSE,
                FsmState.FINAL));
        return new CocaineMethodApi(
            DefaultCocaineProtocolRegistry.Protocols.STREAMING,
            initialState,
            initialState);
    }

    @Override
    protected CocaineSession sessionFor(final Message message) {
        CocaineSession session = super.sessionFor(message);
        if (session == null && message.getType() == INVOKE_METHOD_ID) {
            long sessionId = message.getSession();
            synchronized (this) {
                if (sessionId > maxSessionId) {
                    maxSessionId = sessionId;
                } else {
                    return null;
                }
            }
            session = new CocaineWorkerSession(
                new CocaineSessionContext(sessionId, this, INVOKE_METHOD_API),
                eventHandlerRegistry,
                message.getHeaders());
            Long sessionIdBoxed = sessionId;
            synchronized (sessions) {
                sessions.put(sessionIdBoxed, session);
            }
        }
        return session;
    }
}

