package ru.yandex.client.cocaine.worker;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;

import cocaine.hpack.HeaderField;
import org.msgpack.type.RawValue;
import org.msgpack.type.Value;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineProtocolException;
import ru.yandex.client.cocaine.CocaineServiceException;
import ru.yandex.client.cocaine.CocaineSession;
import ru.yandex.client.cocaine.CocaineSessionContext;
import ru.yandex.client.cocaine.protocol.DefaultCocaineProtocolRegistry;

public class CocaineWorkerSession extends CocaineSession {
    private final long bornDate = System.currentTimeMillis();
    private final CocaineEventHandlerRegistry eventHandlerRegistry;
    private final List<HeaderField> headers;
    private CocaineEventHandler eventHandler = null;
    private CocaineEventConsumer eventConsumer = null;

    public CocaineWorkerSession(
        final CocaineSessionContext context,
        final CocaineEventHandlerRegistry eventHandlerRegistry,
        final List<HeaderField> headers)
    {
        super(context);
        this.eventHandlerRegistry = eventHandlerRegistry;
        this.headers = headers;
    }

    public long bornDate() {
        return bornDate;
    }

    public List<HeaderField> headers() {
        return headers;
    }

    @Override
    protected void failed(final CocaineException e) {
        if (eventConsumer != null) {
            eventConsumer.failed(e);
        }
    }

    private static RawValue asRawValue(final Value value)
        throws CocaineProtocolException
    {
        if (value.isRawValue()) {
            return value.asRawValue();
        } else {
            throw new CocaineProtocolException(value);
        }
    }

    @Override
    protected void payloadReceived(
        final String messageType,
        final Value payload,
        final List<HeaderField> headers)
        throws CocaineException
    {
        if (eventConsumer == null) {
            if (eventHandler == null) {
                if (DefaultCocaineProtocolRegistry.WRITE.equals(messageType)) {
                    String eventName = asRawValue(payload).getString();
                    eventHandler =
                        eventHandlerRegistry.lookup(eventName, headers);
                    if (eventHandler == null) {
                        try {
                            sendMessage(
                                CocaineWorkerService.ERROR_MESSAGE_ID,
                                Arrays.asList(
                                    Arrays.asList(
                                        CocaineServiceException.Category
                                            .FRAMEWORK,
                                        CocaineServiceException.Code
                                            .NO_HANDLER),
                                    "No handler for event '" + eventName
                                    + "' with headers " + headers));
                            close();
                        } catch (IOException e) {
                            throw new CocaineException(e);
                        }
                    }
                } else {
                    throw new CocaineProtocolException(messageType, payload);
                }
            } else {
                if (DefaultCocaineProtocolRegistry.WRITE.equals(messageType)) {
                    eventConsumer = eventHandler.handle(
                        asRawValue(payload),
                        headers,
                        this);
                } else {
                    throw new CocaineProtocolException(messageType, payload);
                }
            }
        } else if (DefaultCocaineProtocolRegistry.WRITE.equals(messageType)) {
            eventConsumer.accept(asRawValue(payload), headers);
        } else if (DefaultCocaineProtocolRegistry.CLOSE.equals(messageType)) {
            eventConsumer.eof();
            // No other cases possible, as 'error' is already handler by
            // protocol handler
        }
    }
}

