package ru.yandex.client.cocaine.worker;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Executor;

import cocaine.hpack.HeaderField;
import org.msgpack.type.Value;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineSession;
import ru.yandex.client.cocaine.CocaineSessionContext;
import ru.yandex.client.cocaine.CocaineSessionFactory;
import ru.yandex.client.cocaine.protocol.DefaultCocaineProtocolRegistry;

public class CocaineWorkerSystemSession extends CocaineSession {
    private final Timer shutdownTimer = new Timer("ShutdownTimer", true);
    private final Timer heartbeatTimer = new Timer("HeartbeatTimer", true);
    private final Context context;
    private final Runnable heartbeat;
    private long lastHeartbeat = System.currentTimeMillis();
    private ShutdownTask shutdownTask;

    public CocaineWorkerSystemSession(
        final CocaineSessionContext cocaineSessionContext,
        final Context context)
    {
        super(cocaineSessionContext);
        this.context = context;
        heartbeat = new Runnable() {
            @Override
            public void run() {
                rescheduleShutdown(
                    context.workerConfig().heartbeatReplyTimeout());
                lastHeartbeat = System.currentTimeMillis();
                try {
                    sendMessage(
                        CocaineWorkerService.HEARTBEAT_MESSAGE_ID,
                        Collections.emptyList());
                } catch (IOException e) {
                    heartbeatSendFailed(e);
                }
            }
        };
        shutdownTask = new ShutdownTask(this);
    }

    @Override
    protected void failed(final CocaineException e) {
    }

    @Override
    protected void payloadReceived(
        final String messageType,
        final Value payload,
        final List<HeaderField> headers)
    {
        switch (messageType) {
            case DefaultCocaineProtocolRegistry.HEARTBEAT:
                scheduleHeartbeat(context.workerConfig().heartbeatInterval());
                break;
            case DefaultCocaineProtocolRegistry.TERMINATE:
                close();
                break;
            default:
                break;
        }
    }

    @Override
    public void close() {
        super.close();
        context.terminateCallback().run();
        heartbeatTimer.cancel();
    }

    protected void heartbeatSendFailed(final IOException e) {
    }

    private void rescheduleShutdown(final long delay) {
        synchronized (this) {
            shutdownTask.cancel();
            shutdownTask = new ShutdownTask(this);
            shutdownTimer.schedule(shutdownTask, delay);
        }
    }

    public void scheduleHeartbeat(final long maxDelay) {
        HeartbeatTask task = new HeartbeatTask(context.executor(), heartbeat);
        long delay = lastHeartbeat + maxDelay - System.currentTimeMillis();
        if (delay <= 0) {
            rescheduleShutdown(
                context.workerConfig().heartbeatExecutionTimeout());
            task.run();
        } else {
            rescheduleShutdown(
                context.workerConfig().heartbeatExecutionTimeout() + delay);
            heartbeatTimer.schedule(task, delay);
        }
    }

    public static class Context {
        private final Runnable terminateCallback;
        private final Executor executor;
        private final ImmutableCocaineWorkerServiceConfig workerConfig;

        public Context(
            final Runnable terminateCallback,
            final Executor executor,
            final ImmutableCocaineWorkerServiceConfig workerConfig)
        {
            this.terminateCallback = terminateCallback;
            this.executor = executor;
            this.workerConfig = workerConfig;
        }

        public Runnable terminateCallback() {
            return terminateCallback;
        }

        public Executor executor() {
            return executor;
        }

        public ImmutableCocaineWorkerServiceConfig workerConfig() {
            return workerConfig;
        }
    }

    public static class Factory
        implements CocaineSessionFactory<CocaineWorkerSystemSession>
    {
        private final Context cocaineWorkerSessionContext;

        public Factory(
            final Runnable terminateCallback,
            final Executor executor,
            final ImmutableCocaineWorkerServiceConfig workerConfig)
        {
            this(new Context(terminateCallback, executor, workerConfig));
        }

        public Factory(final Context cocaineWorkerSessionContext) {
            this.cocaineWorkerSessionContext = cocaineWorkerSessionContext;
        }

        @Override
        public CocaineWorkerSystemSession createSession(
            final CocaineSessionContext cocaineSessionContext)
        {
            return new CocaineWorkerSystemSession(
                cocaineSessionContext,
                cocaineWorkerSessionContext);
        }
    }

    private static class HeartbeatTask extends TimerTask {
        private final Executor executor;
        private final Runnable heartbeat;

        HeartbeatTask(final Executor executor, final Runnable heartbeat) {
            this.executor = executor;
            this.heartbeat = heartbeat;
        }

        @Override
        public void run() {
            executor.execute(heartbeat);
        }
    }

    private static class ShutdownTask extends TimerTask {
        private final CocaineWorkerSystemSession session;

        ShutdownTask(final CocaineWorkerSystemSession session) {
            this.session = session;
        }

        @Override
        public void run() {
            session.close();
        }
    }
}

