package ru.yandex.tikaite.srw;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import cocaine.hpack.HeaderField;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;
import org.apache.tika.io.TikaInputStream;
import org.apache.tika.mime.MediaType;

import ru.yandex.charset.StreamEncoder;
import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineServiceException;
import ru.yandex.client.cocaine.logging.CocaineLoggingService;
import ru.yandex.client.cocaine.unistorage.UnistorageInputStream;
import ru.yandex.client.cocaine.unistorage.UnistorageService;
import ru.yandex.client.cocaine.unistorage.UnistorageSession;
import ru.yandex.client.cocaine.worker.CocaineWorkerSession;
import ru.yandex.client.cocaine.worker.http.AbstractCocaineHttpRequestHandler;
import ru.yandex.client.cocaine.worker.http.BufferedCocaineHttpOutputStreamCloser;
import ru.yandex.client.cocaine.worker.http.CocaineHttpOutputStream;
import ru.yandex.client.cocaine.worker.http.CocaineServiceExceptionConverter;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.UnsupportedMediaTypeException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.tikaite.server.DiskHandler;
import ru.yandex.tikaite.util.DetectionResult;
import ru.yandex.tikaite.util.ExtractionFinalizer;
import ru.yandex.tikaite.util.TextExtractOptions;
import ru.yandex.tikaite.util.TextExtractor;

public class SrwDiskHandler extends AbstractCocaineHttpRequestHandler {
    private final UnistorageService unistorage;
    private final TikaiteSrw tikaiteSrw;

    public SrwDiskHandler(
        final CocaineLoggingService logging,
        final UnistorageService unistorage,
        final TikaiteSrw tikaiteSrw)
    {
        super(logging, tikaiteSrw.config().minRequestTime());
        this.unistorage = unistorage;
        this.tikaiteSrw = tikaiteSrw;
    }

    // CSOFF: ParameterNumber
    @Override
    protected void handle(
        final HttpRequest request,
        final List<HeaderField> headers,
        final CocaineWorkerSession session,
        final Logger logger)
        throws HttpException
    {
        CgiParams params = new CgiParams(request);
        String stid = params.get("stid", NonEmptyValidator.INSTANCE);
        logger.info("Requesting stid: " + stid + " from " + unistorage);
        long start = System.currentTimeMillis();
        try (UnistorageSession unistorageSession =
                unistorage.read(stid, session.headers()))
        {
            unistorageSession.requestMeta();
            long contentLength = unistorageSession.get().asMeta().size();
            logger.info(
                "Got file size " + contentLength + " on " + session
                + " in " + (System.currentTimeMillis() - start) + " ms");
            TextExtractOptions options = DiskHandler.extractOptions(
                params,
                contentLength,
                tikaiteSrw);
            try (TikaInputStream in = TikaInputStream.get(
                    new UnistorageInputStream(unistorageSession),
                    options.createTemporaryResources(),
                    contentLength))
            {
                DetectionResult dr =
                    TextExtractor.INSTANCE.detectStreamType(in, options);
                logger.info(
                    "Mimetype detected: " + dr.mediaType()
                    + ", parser: " + dr.parser());
                if (dr.mediaType().equals(MediaType.OCTET_STREAM)) {
                    throw new UnsupportedMediaTypeException();
                }
                JsonType jsonType = JsonTypeExtractor.NORMAL.extract(params);
                Charset charset = CharsetUtils.acceptedCharset(request);
                try (CocaineHttpOutputStream httpOutput =
                        new CocaineHttpOutputStream(
                            session,
                            HttpStatus.SC_OK,
                            Collections.singletonList(
                                new BasicHeader(
                                    HttpHeaders.CONTENT_TYPE,
                                    ContentType.APPLICATION_JSON
                                        .withCharset(charset).toString()))))
                {
                    try {
                        JsonWriter out = jsonType.create(
                            new StreamEncoder(
                                new BufferedCocaineHttpOutputStreamCloser(
                                    httpOutput),
                                charset.newEncoder()
                                    .onMalformedInput(
                                        CodingErrorAction.REPLACE)
                                    .onUnmappableCharacter(
                                        CodingErrorAction.REPLACE)));
                        new DiskHandler.TextProducer(
                            new ExtractionFinalizer(dr, dr, logger),
                            options)
                            .writeValue(out);
                        out.close();
                    } catch (Throwable t) {
                        logger.log(Level.WARNING, "Response send failed", t);
                        StringBuilderWriter sbw = new StringBuilderWriter();
                        sbw.write("Failed to send response: ");
                        t.printStackTrace(sbw);
                        httpOutput.sendError(
                            CocaineServiceException.Category.FRAMEWORK,
                            CocaineServiceException.Code.INVOCATION_FAILED,
                            sbw.toString());
                    }
                }
            }
        } catch (CocaineServiceException e) {
            throw CocaineServiceExceptionConverter.INSTANCE.apply(e);
        } catch (IOException e) {
            unwrapCocaineServiceException(e);
            throw new BadGatewayException(e);
        } catch (CocaineException | InterruptedException e) {
            throw new BadGatewayException(e);
        }
    }
    // CSON: ParameterNumber
}

