package ru.yandex.tikaite.srw;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.PublicKey;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.LongAdder;
import java.util.logging.Level;
import java.util.logging.Logger;

import cocaine.hpack.HeaderField;
import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;

import ru.yandex.charset.StreamEncoder;
import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineServiceException;
import ru.yandex.client.cocaine.logging.CocaineLoggingService;
import ru.yandex.client.cocaine.unistorage.UnistorageInputStream;
import ru.yandex.client.cocaine.unistorage.UnistorageService;
import ru.yandex.client.cocaine.worker.CocaineWorkerSession;
import ru.yandex.client.cocaine.worker.http.AbstractCocaineHttpRequestHandler;
import ru.yandex.client.cocaine.worker.http.BufferedCocaineHttpOutputStreamCloser;
import ru.yandex.client.cocaine.worker.http.CocaineHttpOutputStream;
import ru.yandex.client.cocaine.worker.http.CocaineServiceExceptionConverter;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.CollectionConsumer;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.so.factors.BasicSoFunctionInputs;
import ru.yandex.mail.so.factors.LoggingFactorsAccessViolationHandler;
import ru.yandex.mail.so.factors.types.BinarySoFactorType;
import ru.yandex.mail.so.factors.types.SmtpEnvelopeSoFactorType;
import ru.yandex.parser.mail.envelope.SmtpEnvelopeHolder;
import ru.yandex.parser.mail.errors.ErrorInfo;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryParser;
import ru.yandex.parser.uri.UriParser;

public class SrwExtractHandler extends AbstractCocaineHttpRequestHandler {
    private static final byte[] SKIP_MARK =
        "\n</message>\n".getBytes(StandardCharsets.UTF_8);

    private final UnistorageService unistorage;
    private final TikaiteSrw tikaiteSrw;
    private final PublicKey rsaPublicKey;

    public SrwExtractHandler(
        final CocaineLoggingService logging,
        final UnistorageService unistorage,
        final TikaiteSrw tikaiteSrw)
    {
        super(logging, tikaiteSrw.config().minRequestTime());
        this.unistorage = unistorage;
        this.tikaiteSrw = tikaiteSrw;
        rsaPublicKey = tikaiteSrw.config().publicKey();
    }

    @Override
    protected void handle(
        final HttpRequest request,
        final List<HeaderField> headers,
        final CocaineWorkerSession session,
        final Logger logger)
        throws HttpException
    {
        LoggingFactorsAccessViolationHandler accessViolationHandler =
            new LoggingFactorsAccessViolationHandler(new LongAdder(), logger);
        List<ErrorInfo> errors = new ArrayList<>();
        CollectionConsumer<ErrorInfo> errorsConsumer =
            new CollectionConsumer<>(errors);
        CocaineSoFactorsExtractorContext extractorContext =
            new CocaineSoFactorsExtractorContext(
                new PrefixedLogger(logger, "", "\t"),
                accessViolationHandler,
                errorsConsumer);
        QueryParser queryParser =
            new UriParser(request.getRequestLine().getUri()).queryParser();
        CgiParams params = new CgiParams(queryParser);
        SmtpEnvelopeHolder envelope =
            new SmtpEnvelopeHolder(errorsConsumer, params);
        EncryptionContext encryptionContext;
        if (params.getBoolean("encrypt", false)) {
            encryptionContext = new RsaEncryptionContext(rsaPublicKey);
        } else {
            encryptionContext = EmptyEncryptionContext.INSTANCE;
        }

        String stid = params.get("stid", NonEmptyValidator.INSTANCE);
        Charset charset = CharsetUtils.acceptedCharset(request);
        logger.info("Requesting stid: " + stid + " from " + unistorage);
        try (InputStream is = new UnistorageInputStream(
                unistorage.read(stid, session.headers()));
            BufferedInputStream in = new BufferedInputStream(is))
        {
            in.mark(1);
            int first = in.read();
            in.reset();
            if (first == '<') {
                IOStreamUtils.skipTo(in, SKIP_MARK);
            }
            ByteArrayProcessable body =
                IOStreamUtils.consume(in).toByteArrayProcessable();
            String extractorName = params.getString("extractor-name", "main");
            Callback callback = new Callback(logger);
            tikaiteSrw.extractors().extract(
                extractorName,
                extractorContext,
                new BasicSoFunctionInputs(
                    accessViolationHandler,
                    BinarySoFactorType.RAW_MAIL.createFactor(body),
                    SmtpEnvelopeSoFactorType.SMTP_ENVELOPE.createFactor(
                        envelope)),
                JsonTypeExtractor.NORMAL.extract(params),
                callback);

            List<Header> responseHeaders = new ArrayList<>(3);
            responseHeaders.add(
                new BasicHeader(
                    HttpHeaders.CONTENT_TYPE,
                    ContentType.APPLICATION_JSON
                        .withCharset(charset).toString()));
            encryptionContext.adjustResponseHeaders(responseHeaders);
            try (CocaineHttpOutputStream httpOutput =
                    new CocaineHttpOutputStream(
                        session,
                        HttpStatus.SC_OK,
                        responseHeaders))
            {
                try {
                    Writer out =
                        new StreamEncoder(
                            encryptionContext.wrapOutputStream(
                                new BufferedCocaineHttpOutputStreamCloser(
                                    httpOutput)),
                            charset.newEncoder()
                                .onMalformedInput(CodingErrorAction.REPLACE)
                                .onUnmappableCharacter(
                                    CodingErrorAction.REPLACE));
                    out.write(callback.result);
                    out.close();
                } catch (Throwable t) {
                    logger.log(Level.WARNING, "Response send failed", t);
                    StringBuilderWriter sbw = new StringBuilderWriter();
                    sbw.write("Failed to send response: ");
                    t.printStackTrace(sbw);
                    httpOutput.sendError(
                        CocaineServiceException.Category.FRAMEWORK,
                        CocaineServiceException.Code.INVOCATION_FAILED,
                        sbw.toString());
                }
            }
        } catch (CocaineServiceException e) {
            throw CocaineServiceExceptionConverter.INSTANCE.apply(e);
        } catch (IOException e) {
            unwrapCocaineServiceException(e);
            throw new BadGatewayException(e);
        } catch (CocaineException e) {
            throw new BadGatewayException(e);
        } catch (GeneralSecurityException e) {
            throw new ServiceUnavailableException(e);
        }
    }

    private static class Callback implements FutureCallback<String> {
        private final Logger logger;
        private String result = "{}";

        Callback(final Logger logger) {
            this.logger = logger;
        }

        @Override
        public void cancelled() {
            logger.warning("Request cancelled");
        }

        @Override
        public void failed(final Exception e) {
            logger.log(Level.WARNING, "Processing failed", e);
        }

        @Override
        public void completed(final String result) {
            this.result = result;
        }
    }
}

