package ru.yandex.tikaite.srw;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import cocaine.hpack.HeaderField;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.james.mime4j.stream.Field;

import ru.yandex.client.cocaine.logging.CocaineLoggingService;
import ru.yandex.client.cocaine.unistorage.UnistorageService;
import ru.yandex.client.cocaine.worker.CocaineWorkerSession;
import ru.yandex.client.cocaine.worker.http.AbstractCocaineHttpRequestHandler;
import ru.yandex.client.cocaine.worker.http.CocaineDummyHttpRequestHandler;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.util.storage.DataExtractorConfigBuilder;
import ru.yandex.util.storage.ImmutableDataExtractorConfig;
import ru.yandex.util.storage.MailTextExtractor;
import ru.yandex.util.storage.cocaine.CocaineStorageClient;

public class SrwHeadersHandler extends AbstractCocaineHttpRequestHandler {
    private static final ImmutableDataExtractorConfig DATA_EXTRACTOR_CONFIG =
        new DataExtractorConfigBuilder().build();
    private static final int START_HEADERS_SIZE = 30;

    private final UnistorageService unistorage;

    public SrwHeadersHandler(
        final CocaineLoggingService logging,
        final UnistorageService unistorage)
    {
        super(logging, 0L);
        this.unistorage = unistorage;
    }

    @Override
    protected void handle(
        final HttpRequest request,
        final List<HeaderField> headers,
        final CocaineWorkerSession session,
        final Logger logger)
        throws HttpException
    {
        Charset acceptedCharset = CharsetUtils.acceptedCharset(request);
        CgiParams params = new CgiParams(request);
        String stid = params.get("stid", NonEmptyValidator.INSTANCE);
        JsonType jsonType = JsonTypeExtractor.NORMAL.extract(params);
        HeadersExtractor headersExtractor = new HeadersExtractor();
        new CocaineStorageClient(
            unistorage,
            headersExtractor,
            DATA_EXTRACTOR_CONFIG)
            .sendStorageRequest(stid, session, logger);
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter jsonWriter = jsonType.create(sbw)) {
            jsonWriter.startObject();
            jsonWriter.key("headers");
            jsonWriter.startArray();
            if (headersExtractor.headers != null) {
                for (Map.Entry<String, String> header:
                    headersExtractor.headers)
                {
                    jsonWriter.startObject();
                    jsonWriter.key(header.getKey());
                    jsonWriter.value(header.getValue());
                    jsonWriter.endObject();
                }
            }
            jsonWriter.endArray();
            jsonWriter.endObject();
        } catch (IOException e) {
            throw new ServiceUnavailableException(e);
        }
        new CocaineDummyHttpRequestHandler(
            HttpStatus.SC_OK,
            new StringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON.withCharset(acceptedCharset)))
            .handle(request, headers, session);
    }

    private static class HeadersExtractor extends MailTextExtractor {
        List<Map.Entry<String, String>> headers = null;

        @Override
        protected void processField(final String name, final Field field) {
            if (headers == null) {
                headers = new ArrayList<>(START_HEADERS_SIZE);
            }
            headers.add(new SimpleEntry<>(name, field.getBody()));
        }

        @Override
        public boolean findHeadersOnly() {
            return true;
        }
    }
}

