package ru.yandex.tikaite.srw;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.CodingErrorAction;
import java.security.GeneralSecurityException;
import java.security.PublicKey;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import cocaine.hpack.HeaderField;
import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;
import org.apache.james.mime4j.MimeException;

import ru.yandex.charset.StreamEncoder;
import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.CocaineServiceException;
import ru.yandex.client.cocaine.logging.CocaineLoggingService;
import ru.yandex.client.cocaine.unistorage.UnistorageInputStream;
import ru.yandex.client.cocaine.unistorage.UnistorageService;
import ru.yandex.client.cocaine.worker.CocaineWorkerSession;
import ru.yandex.client.cocaine.worker.http.AbstractCocaineHttpRequestHandler;
import ru.yandex.client.cocaine.worker.http.BufferedCocaineHttpOutputStreamCloser;
import ru.yandex.client.cocaine.worker.http.CocaineHttpOutputStream;
import ru.yandex.client.cocaine.worker.http.CocaineServiceExceptionConverter;
import ru.yandex.http.util.BadGatewayException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.UnsupportedMediaTypeException;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryParser;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.search.document.mail.MailMetaInfo;
import ru.yandex.tikaite.mimeparser.EnclosedParser;
import ru.yandex.tikaite.server.OfflineMailHandler;
import ru.yandex.tikaite.server.OnlineMailHandler;
import ru.yandex.tikaite.util.TextExtractOptions;

public class SrwMailHandler extends AbstractCocaineHttpRequestHandler {
    private final UnistorageService unistorage;
    private final TikaiteSrw tikaiteSrw;
    private final PublicKey rsaPublicKey;

    public SrwMailHandler(
        final CocaineLoggingService logging,
        final UnistorageService unistorage,
        final TikaiteSrw tikaiteSrw)
    {
        super(logging, tikaiteSrw.config().minRequestTime());
        this.unistorage = unistorage;
        this.tikaiteSrw = tikaiteSrw;
        rsaPublicKey = tikaiteSrw.config().publicKey();
    }

    @Override
    protected void handle(
        final HttpRequest request,
        final List<HeaderField> headers,
        final CocaineWorkerSession session,
        final Logger logger)
        throws HttpException
    {
        QueryParser queryParser =
            new UriParser(request.getRequestLine().getUri()).queryParser();
        CgiParams params = new CgiParams(queryParser);
        EncryptionContext encryptionContext;
        if (params.getBoolean("encrypt", false)) {
            encryptionContext = new RsaEncryptionContext(rsaPublicKey);
        } else {
            encryptionContext = EmptyEncryptionContext.INSTANCE;
        }
        String stid = params.get("stid", NonEmptyValidator.INSTANCE);
        logger.info("Requesting stid: " + stid + " from " + unistorage);
        long start = System.currentTimeMillis();
        TextExtractOptions options =
            OnlineMailHandler.extractOptions(params, tikaiteSrw);
        MailMetaInfo meta = new MailMetaInfo(
            options.headersLimit(),
            options.headerLimit(),
            tikaiteSrw.yandexNets());
        OfflineMailHandler.extractMeta(meta, queryParser);
        logger.fine("Meta information extracted");
        JsonType jsonType = JsonTypeExtractor.NORMAL.extract(params);
        Charset charset = CharsetUtils.acceptedCharset(request);
        try (InputStream in = new UnistorageInputStream(
                unistorage.read(stid, session.headers()));
            EnclosedParser parser = new EnclosedParser(meta, in, options))
        {
            logger.info(
                "Main headers parsed in "
                + (System.currentTimeMillis() - start)
                + " ms");
            List<Header> responseHeaders = new ArrayList<>(3);
            responseHeaders.add(
                new BasicHeader(
                    HttpHeaders.CONTENT_TYPE,
                    ContentType.APPLICATION_JSON
                        .withCharset(charset).toString()));
            encryptionContext.adjustResponseHeaders(responseHeaders);
            try (CocaineHttpOutputStream httpOutput =
                    new CocaineHttpOutputStream(
                        session,
                        HttpStatus.SC_OK,
                        responseHeaders))
            {
                try {
                    JsonWriter out =
                        jsonType.create(
                            new StreamEncoder(
                                encryptionContext.wrapOutputStream(
                                    new BufferedCocaineHttpOutputStreamCloser(
                                        httpOutput)),
                                charset.newEncoder()
                                    .onMalformedInput(
                                        CodingErrorAction.REPLACE)
                                    .onUnmappableCharacter(
                                        CodingErrorAction.REPLACE)));
                    parser.parse(out);
                    out.close();
                } catch (Throwable t) {
                    logger.log(Level.WARNING, "Response send failed", t);
                    StringBuilderWriter sbw = new StringBuilderWriter();
                    sbw.write("Failed to send response: ");
                    t.printStackTrace(sbw);
                    httpOutput.sendError(
                        CocaineServiceException.Category.FRAMEWORK,
                        CocaineServiceException.Code.INVOCATION_FAILED,
                        sbw.toString());
                }
            }
        } catch (EOFException | MimeException e) {
            throw new UnsupportedMediaTypeException(
                "Failed to extract meta",
                e);
        } catch (CocaineServiceException e) {
            throw CocaineServiceExceptionConverter.INSTANCE.apply(e);
        } catch (IOException e) {
            unwrapCocaineServiceException(e);
            throw new BadGatewayException(e);
        } catch (CocaineException e) {
            throw new BadGatewayException(e);
        } catch (GeneralSecurityException e) {
            throw new ServiceUnavailableException(e);
        }
    }
}

