package ru.yandex.tikaite.srw;

import java.io.IOException;
import java.nio.file.Paths;
import java.util.concurrent.atomic.LongAdder;

import ru.yandex.client.cocaine.CocaineException;
import ru.yandex.client.cocaine.worker.CocaineWorkerConfig;
import ru.yandex.client.cocaine.worker.http.CocaineExecutorHttpRequestHandler;
import ru.yandex.client.cocaine.worker.http.CocaineHttpEventHandler;
import ru.yandex.client.cocaine.worker.http.unistorage.UnistorageHttpService;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.ConstFunction;
import ru.yandex.function.EmptyRunnable;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.function.GenericNonThrowingCloseableAdapter;
import ru.yandex.function.NullConsumer;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.logger.DevNullLogger;
import ru.yandex.mail.so.factors.config.ImmutableExtractModulesConfig;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorModules;
import ru.yandex.mail.so.factors.extractors.SoFactorsExtractorsRegistry;
import ru.yandex.mail.so.factors.types.SoFactorTypesRegistry;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.sanitizer2.PageHeaderException;
import ru.yandex.sanitizer2.SanitizingHandler;
import ru.yandex.sanitizer2.config.ImmutableSanitizingConfig;
import ru.yandex.search.document.DocumentFieldsFilter;
import ru.yandex.search.document.IdentityDocumentFieldsFilter;
import ru.yandex.stater.NullStatersRegistrar;
import ru.yandex.tikaite.config.ImmutableFieldsFilterConfig;
import ru.yandex.tikaite.config.ImmutableTextExtractorConfig;
import ru.yandex.tikaite.server.TikaiteContext;
import ru.yandex.tikaite.util.FieldsFilter;
import ru.yandex.util.ip.ReloadableIpChecker;

public class TikaiteSrw
    extends UnistorageHttpService
    implements TikaiteContext
{
    private final ImmutableTikaiteSrwConfig config;
    private final ImmutableTextExtractorConfig textExtractorConfig;
    private final DocumentFieldsFilter fieldsFilter;
    private final SanitizingHandler sanitizer;
    private final ReloadableIpChecker yandexNets;
    private final SoFactorsExtractorModules extractors;

    public TikaiteSrw(
        final ImmutableTikaiteSrwConfig serviceConfig,
        final CocaineWorkerConfig workerConfig)
        throws CocaineException,
            ConfigException,
            IOException,
            InterruptedException,
            JniWrapperException,
            PageHeaderException
    {
        super(serviceConfig, workerConfig, "TikaiteSrw");
        config = serviceConfig;
        textExtractorConfig = config.textExtractorConfig();
        ImmutableFieldsFilterConfig fieldsFilterConfig =
            textExtractorConfig.fieldsFilterConfig();
        try (GenericAutoCloseableHolder<
                IOException,
                GenericAutoCloseableChain<IOException>> chain =
                    new GenericAutoCloseableHolder<>(this.chain))
        {
            if (fieldsFilterConfig == null) {
                fieldsFilter = IdentityDocumentFieldsFilter.INSTANCE;
            } else {
                FieldsFilter fieldsFilter = new FieldsFilter(
                    fieldsFilterConfig,
                    new ThreadFactoryConfig("FieldsFilter-")
                        .group(Thread.currentThread().getThreadGroup())
                        .daemon(true)
                        .uncaughtExceptionHandler(this));
                chain.get().add(
                    new GenericNonThrowingCloseableAdapter<>(fieldsFilter));
                this.fieldsFilter = fieldsFilter;
            }
            ImmutableSanitizingConfig sanitizingConfig =
                textExtractorConfig.sanitizingConfig();
            if (sanitizingConfig == null) {
                sanitizer = null;
            } else {
                sanitizer = new SanitizingHandler(
                    sanitizingConfig,
                    NullConsumer.instance(),
                    EmptyRunnable.INSTANCE);
                chain.get().add(
                    new GenericNonThrowingCloseableAdapter<>(sanitizer));
                handlerRegistry.register(
                    "sanitizer",
                    new CocaineHttpEventHandler(
                        new CocaineExecutorHttpRequestHandler(
                            executor,
                            new SrwSanitizerHandler(
                                logging,
                                unistorage,
                                sanitizer)),
                        serviceConfig.readTimeout()));
            }
            yandexNets = textExtractorConfig
                .receivedChainParserConfig()
                .yandexNetsConfig()
                .createIpChecker();
            ImmutableExtractModulesConfig extractModulesConfig =
                serviceConfig.extractModulesConfig();
            if (extractModulesConfig == null) {
                extractors = null;
            } else {
                SoFactorsExtractorsRegistry extractorsRegistry =
                    new SoFactorsExtractorsRegistry(
                        NullStatersRegistrar.INSTANCE,
                        new SoFactorTypesRegistry());
                chain.get().add(extractorsRegistry);
                extractors = new SoFactorsExtractorModules(
                    serviceConfig.extractModulesConfig(),
                    new LongAdder(),
                    extractorsRegistry,
                    new ConstFunction<>(NullConsumer.instance()),
                    new LongAdder(),
                    Thread.currentThread().getThreadGroup(),
                    null,
                    null,
                    DevNullLogger.INSTANCE,
                    null,
                    0L);
                chain.get().add(extractors);
            }

            handlerRegistry.register(
                "disk",
                new CocaineHttpEventHandler(
                    new CocaineExecutorHttpRequestHandler(
                        executor,
                        new SrwDiskHandler(logging, unistorage, this)),
                    serviceConfig.readTimeout()));
            CocaineHttpEventHandler mailHandler =
                new CocaineHttpEventHandler(
                    new CocaineExecutorHttpRequestHandler(
                        executor,
                        new SrwMailHandler(logging, unistorage, this)),
                    serviceConfig.readTimeout());
            handlerRegistry.register("mail", mailHandler);
            handlerRegistry.register("ml", mailHandler);
            if (extractors != null) {
                handlerRegistry.register(
                    "extract",
                    new CocaineHttpEventHandler(
                        new CocaineExecutorHttpRequestHandler(
                            executor,
                            new SrwExtractHandler(
                                logging,
                                unistorage,
                                this)),
                        serviceConfig.readTimeout()));
            }
            handlerRegistry.register(
                "text",
                new CocaineHttpEventHandler(
                    new CocaineExecutorHttpRequestHandler(
                        executor,
                        new SrwTextHandler(logging, unistorage, this)),
                    serviceConfig.readTimeout()));
            handlerRegistry.register(
                "headers",
                new CocaineHttpEventHandler(
                    new CocaineExecutorHttpRequestHandler(
                        executor,
                        new SrwHeadersHandler(logging, unistorage)),
                    serviceConfig.readTimeout()));
            chain.release();
        }
    }

    public ImmutableTikaiteSrwConfig config() {
        return config;
    }

    @Override
    public ImmutableTextExtractorConfig textExtractorConfig() {
        return textExtractorConfig;
    }

    @Override
    public DocumentFieldsFilter fieldsFilter() {
        return fieldsFilter;
    }

    @Override
    public SanitizingHandler sanitizer() {
        return sanitizer;
    }

    @Override
    public ReloadableIpChecker yandexNets() {
        return yandexNets;
    }

    public SoFactorsExtractorModules extractors() {
        return extractors;
    }

    public static void main(final String... args)
        throws CocaineException,
            ConfigException,
            IOException,
            InterruptedException,
            JniWrapperException,
            PageHeaderException
    {
        IniConfig config = new IniConfig(Paths.get(args[0]));
        TikaiteSrw server = new TikaiteSrw(
            new ImmutableTikaiteSrwConfig(new TikaiteSrwConfigBuilder(config)),
            new CocaineWorkerConfig(args));
        config.checkUnusedKeys();
        server.start();
    }
}

