package ru.yandex.cokemulator;

import org.apache.http.HttpHost;
import org.apache.http.entity.ContentType;

import ru.yandex.client.tvm2.Tvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.jniwrapper.JniWrapperConfig;
import ru.yandex.jniwrapper.JniWrapperConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.util.storage.DataExtractorConfig;
import ru.yandex.util.storage.DataExtractorConfigBuilder;

public abstract class AbstractCokemulatorConfigBuilder
    <T extends AbstractCokemulatorConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements CokemulatorConfig
{
    private HttpHostConfigBuilder storageConfig;
    private HttpHost avaHost;
    private int concurrency;
    private JniWrapperConfigBuilder jniWrapperConfig;
    private DataType dataType;
    private DataExtractorConfigBuilder dataExtractorConfig;
    private ContentType contentType;
    private String uriSuffix;
    private HttpHostConfigBuilder mulcagateConfig;
    private int cokemulatorPort;
    private HttpHostConfigBuilder tikaiteConfig;
    private Tvm2ClientConfigBuilder tvm2ClientConfig;
    private String apeClientId;
    private String unistorageClientId;
    private String uriPrefix;

    protected AbstractCokemulatorConfigBuilder(final CokemulatorConfig config) {
        super(config);
        storageConfig(config.storageConfig());
        concurrency(config.concurrency());
        jniWrapperConfig(config.jniWrapperConfig());
        dataType(config.dataType());
        dataExtractorConfig(config.dataExtractorConfig());
        contentType(config.contentType());
        uriSuffix(config.uriSuffix());
        mulcagateConfig(config.mulcagateConfig());
        cokemulatorPort(config.cokemulatorPort());
        tikaiteConfig(config.tikaiteConfig());
        tvm2ClientConfig(config.tvm2ClientConfig());
        apeClientId(config.apeClientId());
        unistorageClientId(config.unistorageClientId());
        uriPrefix(config.uriPrefix());
        avaHost(config.avaHost());
    }

    protected AbstractCokemulatorConfigBuilder(
        final IniConfig config,
        final CokemulatorConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        storageConfig =
            STORAGE_CONFIG.extract(config, defaults.storageConfig());
        IniConfig storageSection = config.sectionOrNull("storage");
        if (storageSection != null) {
            avaHost = storageSection.get("ava_host", defaults.avaHost(), HttpHostParser.INSTANCE);
        } else {
            avaHost = defaults.avaHost();
        }

        IniConfig cokemulator = config.section("cokemulator");
        concurrency = CONCURRENCY.extract(cokemulator, defaults.concurrency());
        jniWrapperConfig =
            JNIWRAPPER_CONFIG.extract(config, defaults.jniWrapperConfig());
        dataType = cokemulator.getEnum(
            DataType.class,
            "data-type",
            defaults.dataType());
        dataExtractorConfig = new DataExtractorConfigBuilder(
            cokemulator,
            defaults.dataExtractorConfig());
        contentType = cokemulator.get(
            "content-type",
            defaults.contentType(),
            x -> ContentType.create(x));
        uriPrefix = URI_PREFIX.extract(cokemulator, defaults.uriPrefix());
        uriSuffix = URI_SUFFIX.extract(cokemulator, defaults.uriSuffix());

        switch (dataType) {
            case C2C:
                IniConfig mulcagateSection = config.section("mulcagate");
                mulcagateConfig = new HttpHostConfigBuilder(
                    mulcagateSection,
                    defaults.mulcagateConfig());
                cokemulatorPort = COKEMULATOR_PORT.extract(
                    mulcagateSection,
                    defaults.cokemulatorPort());
                break;
            case C2SRW:
                tvm2ClientConfig = new Tvm2ClientConfigBuilder(
                    config.section("tvm2"),
                    defaults.tvm2ClientConfig());
                apeClientId =
                    APE_CLIENT_ID.extract(config, defaults.apeClientId());
                unistorageClientId = UNISTORAGE_CLIENT_ID.extract(
                    config,
                    defaults.unistorageClientId());
                break;
            default:
                IniConfig tikaiteSection = config.sectionOrNull("tikaite");
                if (tikaiteSection != null) {
                    HttpHostConfig tikaiteDefaults = defaults.tikaiteConfig();
                    if (tikaiteDefaults == null) {
                        tikaiteConfig =
                            new HttpHostConfigBuilder(tikaiteSection);
                    } else {
                        tikaiteConfig = new HttpHostConfigBuilder(
                            tikaiteSection,
                            defaults.tikaiteConfig());
                    }
                }
                break;
        }
    }

    @Override
    public HttpHostConfigBuilder storageConfig() {
        return storageConfig;
    }

    public T storageConfig(final HttpHostConfig storageConfig) {
        if (storageConfig == null) {
            this.storageConfig = null;
        } else {
            this.storageConfig = new HttpHostConfigBuilder(storageConfig);
        }
        return self();
    }

    @Override
    public int concurrency() {
        return concurrency;
    }

    public T concurrency(final int concurrency) {
        this.concurrency = concurrency;
        return self();
    }

    @Override
    public JniWrapperConfigBuilder jniWrapperConfig() {
        return jniWrapperConfig;
    }

    public T jniWrapperConfig(final JniWrapperConfig jniWrapperConfig) {
        this.jniWrapperConfig = new JniWrapperConfigBuilder(jniWrapperConfig);
        return self();
    }

    @Override
    public DataType dataType() {
        return dataType;
    }

    public T dataType(final DataType dataType) {
        this.dataType = dataType;
        return self();
    }

    @Override
    public DataExtractorConfigBuilder dataExtractorConfig() {
        return dataExtractorConfig;
    }

    public T dataExtractorConfig(
        final DataExtractorConfig dataExtractorConfig)
    {
        this.dataExtractorConfig =
            new DataExtractorConfigBuilder(dataExtractorConfig);
        return self();
    }

    @Override
    public ContentType contentType() {
        return contentType;
    }

    public T contentType(final ContentType contentType) {
        this.contentType = contentType;
        return self();
    }

    @Override
    public String uriSuffix() {
        return uriSuffix;
    }

    public T uriSuffix(final String uriSuffix) {
        this.uriSuffix = uriSuffix;
        return self();
    }

    @Override
    public HttpHostConfigBuilder mulcagateConfig() {
        return mulcagateConfig;
    }

    public T mulcagateConfig(final HttpHostConfig mulcagateConfig) {
        if (mulcagateConfig == null) {
            this.mulcagateConfig = null;
        } else {
            this.mulcagateConfig = new HttpHostConfigBuilder(mulcagateConfig);
        }
        return self();
    }

    @Override
    public int cokemulatorPort() {
        return cokemulatorPort;
    }

    public T cokemulatorPort(final int cokemulatorPort) {
        this.cokemulatorPort = cokemulatorPort;
        return self();
    }

    @Override
    public HttpHostConfigBuilder tikaiteConfig() {
        return tikaiteConfig;
    }

    public T tikaiteConfig(final HttpHostConfig tikaiteConfig) {
        if (tikaiteConfig == null) {
            this.tikaiteConfig = null;
        } else {
            this.tikaiteConfig = new HttpHostConfigBuilder(tikaiteConfig);
        }
        return self();
    }

    @Override
    public Tvm2ClientConfigBuilder tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    public T tvm2ClientConfig(final Tvm2ClientConfig tvm2ClientConfig) {
        if (tvm2ClientConfig == null) {
            this.tvm2ClientConfig = null;
        } else {
            this.tvm2ClientConfig =
                new Tvm2ClientConfigBuilder(tvm2ClientConfig);
        }
        return self();
    }

    @Override
    public String apeClientId() {
        return apeClientId;
    }

    public T apeClientId(final String apeClientId) {
        this.apeClientId = apeClientId;
        return self();
    }

    @Override
    public String unistorageClientId() {
        return unistorageClientId;
    }

    public T unistorageClientId(final String unistorageClientId) {
        this.unistorageClientId = unistorageClientId;
        return self();
    }

    @Override
    public String uriPrefix() {
        return uriPrefix;
    }

    public T uriPrefix(final String uriPrefix) {
        this.uriPrefix = uriPrefix;
        return self();
    }

    @Override
    public HttpHost avaHost() {
        return avaHost;
    }

    public T avaHost(final HttpHost avaHost) {
        this.avaHost = avaHost;
        return self();
    }
}

