package ru.yandex.cokemulator;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.charset.CharacterCodingException;
import java.security.GeneralSecurityException;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHeader;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.client.tvm2.ImmutableTvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.SemaphoreLockAdapter;
import ru.yandex.concurrent.ThreadFactoryConfig;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericNonThrowingCloseableAdapter;
import ru.yandex.function.StringBuilderProcessorAdapter;
import ru.yandex.function.StringVoidProcessor;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.UnsupportedMediaTypeException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.http.util.nio.client.BasicRequestsListener;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.jniwrapper.JniWrapper;
import ru.yandex.jniwrapper.JniWrapperException;
import ru.yandex.jniwrapper.JniWrapperUnprocessableInputException;
import ru.yandex.json.parser.JsonException;
import ru.yandex.mulcagate.MulcagateClient;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.PctEncoder;
import ru.yandex.parser.uri.PctEncodingRule;
import ru.yandex.util.storage.DataExtractor;
import ru.yandex.util.storage.ImmutableDataExtractorConfig;
import ru.yandex.util.storage.StorageClient;
import ru.yandex.util.storage.StorageData;
import ru.yandex.util.string.StringUtils;

public class Cokemulator
    extends BaseHttpServer<ImmutableCokemulatorConfig>
    implements HttpRequestHandler
{
    private static final Header[] EMPTY_HEADERS = new Header[0];

    private final GenericAutoCloseableChain<IOException> chain =
        new GenericAutoCloseableChain<>();
    private final ImmutableCokemulatorConfig cokemulatorConfig;
    private final Lock processingLock;
    private final ImmutableDataExtractorConfig extractorConfig;
    private final DataExtractor dataExtractor;
    private final StorageClient storageClient;
    private final List<HttpHost> storageHost;
    private final StorageClient tikaiteClient;
    private final List<HttpHost> tikaiteHost;
    private final JniWrapper jniWrapper;
    private final SharedConnectingIOReactor reactor;
    private final MulcagateClient mulcagateClient;
    private final Tvm2TicketRenewalTask tvm2RenewalTask;
    private final HttpHost avaHost;

    public Cokemulator(final ImmutableCokemulatorConfig cokemulatorConfig)
        throws ConfigException,
            GeneralSecurityException,
            HttpException,
            IOException,
            JsonException,
            URISyntaxException
    {
        super(cokemulatorConfig);
        this.avaHost = cokemulatorConfig.avaHost();
        this.cokemulatorConfig = cokemulatorConfig;
        processingLock = SemaphoreLockAdapter.create(
            cokemulatorConfig.concurrency(),
            cokemulatorConfig.workers());
        extractorConfig = cokemulatorConfig.dataExtractorConfig();
        DataType dataType = cokemulatorConfig.dataType();
        dataExtractor = dataType.dataExtractor();
        String additionalCgis = dataType.additionalCgis();
        String uriSuffix = cokemulatorConfig.uriSuffix();
        if (additionalCgis != null) {
            if (uriSuffix == null) {
                uriSuffix = additionalCgis;
            } else {
                uriSuffix = StringUtils.concat(additionalCgis, '&', uriSuffix);
            }
        }
        ImmutableHttpHostConfig storageConfig =
            cokemulatorConfig.storageConfig();
        if (storageConfig == null) {
            storageClient = null;
            storageHost = null;
        } else {
            storageClient = new StorageClient(
                ClientBuilder.createClient(
                    cokemulatorConfig.storageConfig(),
                    cokemulatorConfig.dnsConfig()),
                uriSuffix);
            chain.add(storageClient);
            storageHost = Collections.singletonList(
                cokemulatorConfig.storageConfig().host());
        }
        ImmutableHttpHostConfig tikaiteConfig =
            cokemulatorConfig.tikaiteConfig();
        if (tikaiteConfig == null) {
            tikaiteClient = null;
            tikaiteHost = null;
        } else {
            tikaiteClient = new StorageClient(
                ClientBuilder.createClient(
                    tikaiteConfig,
                    cokemulatorConfig.dnsConfig()),
                null);
            chain.add(tikaiteClient);
            tikaiteHost = Collections.singletonList(tikaiteConfig.host());
        }
        try {
            jniWrapper = JniWrapper.create(
                cokemulatorConfig.jniWrapperConfig(),
                new ThreadFactoryConfig(cokemulatorConfig.name() + "-Jni-")
                    .group(getThreadGroup())
                    .daemon(true));
        } catch (JniWrapperException e) {
            throw new ConfigException("Library loading failed", e);
        }
        chain.add(new GenericNonThrowingCloseableAdapter<>(jniWrapper));
        ImmutableHttpHostConfig mulcagateConfig =
            cokemulatorConfig.mulcagateConfig();
        if (mulcagateConfig == null) {
            reactor = null;
            mulcagateClient = null;
        } else {
            reactor = new SharedConnectingIOReactor(
                cokemulatorConfig,
                cokemulatorConfig.dnsConfig(),
                new ThreadGroup(
                    getThreadGroup(),
                    cokemulatorConfig.name() + "-Client"));
            chain.add(reactor);
            mulcagateClient = new MulcagateClient(reactor, mulcagateConfig);
            chain.add(mulcagateClient);
        }

        ImmutableTvm2ClientConfig tvm2ClientConfig =
            cokemulatorConfig.tvm2ClientConfig();
        if (tvm2ClientConfig == null) {
            tvm2RenewalTask = null;
        } else {
            tvm2RenewalTask = new Tvm2TicketRenewalTask(
                logger().addPrefix("tvm2"),
                serviceContextRenewalTask,
                tvm2ClientConfig);
        }
        register(new Pattern<>("", true), this, RequestHandlerMapper.POST);
        if (storageClient != null) {
            register(
                new Pattern<>("/process", false),
                this,
                RequestHandlerMapper.GET);
            register(
                new Pattern<>("/get-text", false),
                new GetTextHandler(this),
                RequestHandlerMapper.GET);
        }
        if (cokemulatorConfig.jniWrapperConfig().reloadName() != null) {
            register(
                new Pattern<>("/reload", false),
                new ReloadHandler(jniWrapper),
                RequestHandlerMapper.GET);
        }
    }

    public StorageClient storageClient() {
        return storageClient;
    }

    private static void encode(
        final StringVoidProcessor<char[], CharacterCodingException> encoder,
        final String string,
        final StringBuilderProcessorAdapter query)
        throws BadRequestException
    {
        try {
            encoder.process(string);
        } catch (CharacterCodingException e) {
            throw new BadRequestException("Failed to encode " + string, e);
        }
        encoder.processWith(query);
    }

    @SuppressWarnings("StringSplitter")
    protected String createQuery(final String stid, final String hid)
        throws BadRequestException
    {
        StringBuilder query = new StringBuilder();
        if (avaHost != null && isAvaStid(stid)) {
            String strippedStid = stid;
            if (stid.startsWith("/")) {
                strippedStid  = stid.substring(1);
            }
            String[] split = strippedStid.split("/");
            String avatarsPrefix = "avatars-";
            if (split[0].startsWith("get-")) {
                query.append('/');
                query.append(split[0]);
                query.append('/');
            } else if (split[0].startsWith(avatarsPrefix)
                && split[0].length() > avatarsPrefix.length())
            {
                query.append("/get-");
                query.append(split[0].substring(avatarsPrefix.length()));
                query.append('/');
            }

            if (split.length < 4) {
                throw new BadRequestException("Invalid stid " + stid);
            }

            query.append(split[1]);
            query.append('/');
            query.append(split[2]);
            query.append("/");
            query.append(split[3]);
        } else {

            if (cokemulatorConfig.uriPrefix() != null) {
                query.append(cokemulatorConfig.uriPrefix());
            } else {
                if (mulcagateClient == null && tvm2RenewalTask == null) {
                    if (hid == null) {
                        query.append("/get/");
                    } else {
                        query.append("/text?encoding=auto&stid=");
                    }
                } else {
                    query.append("/get-text?stid=");
                }
            }

            PctEncodingRule pctEncodingRule;
            if (mulcagateClient == null && hid == null && tvm2RenewalTask == null) {
                pctEncodingRule = PctEncodingRule.PATH;
            } else {
                pctEncodingRule = PctEncodingRule.QUERY;
            }
            StringVoidProcessor<char[], CharacterCodingException> encoder =
                new StringVoidProcessor<>(new PctEncoder(pctEncodingRule));
            StringBuilderProcessorAdapter adapter =
                new StringBuilderProcessorAdapter(query);
            encode(encoder, stid, adapter);
            if (hid != null) {
                query.append("&hid=");
                encode(encoder, hid, adapter);
            }
        }
        return new String(query);
    }

    public static boolean isAvaStid(final String stid) {
        return stid.contains("/");
    }

    protected List<HttpHost> hostsFor(
        final String stid,
        final String hid,
        final Logger logger)
        throws HttpException
    {
        List<HttpHost> hosts;
        if (mulcagateClient == null) {
            if (hid == null) {
                if (avaHost != null && isAvaStid(stid)) {
                    hosts = Collections.singletonList(avaHost);
                } else {
                    hosts = storageHost;
                }

            } else {
                hosts = tikaiteHost;
            }
        } else {
            logger.fine("Resolving hosts for stid " + stid);
            hosts =
                MulcagateClient.toHttpHosts(
                    mulcagateClient.resolvePrimaryStidHostsSyncronously(
                        stid,
                        new BasicRequestsListener(),
                        logger),
                    cokemulatorConfig.cokemulatorPort());
            Collections.rotate(hosts, stid.hashCode());
        }
        return hosts;
    }

    public StorageData sendStorageRequest(
        final HttpRequest incomingRequest,
        final HttpContext context)
        throws HttpException
    {
        Logger logger = (Logger) context.getAttribute(LOGGER);
        CgiParams params = new CgiParams(incomingRequest);
        String stid = params.get("stid", NonEmptyValidator.INSTANCE);
        String hid;
        if (mulcagateClient == null && tikaiteClient == null) {
            hid = null;
        } else {
            hid = params.get("hid", null, NonEmptyValidator.INSTANCE);
        }
        String query = createQuery(stid, hid);
        Header[] headers;
        if (tvm2RenewalTask == null) {
            headers = EMPTY_HEADERS;
        } else {
            headers = new Header[] {
                new BasicHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    tvm2RenewalTask.ticket(cokemulatorConfig.apeClientId())),
                new BasicHeader(
                    YandexHeaders.X_SRW_SERVICE_TICKET,
                    tvm2RenewalTask.ticket(
                        cokemulatorConfig.unistorageClientId()))
            };
        }
        HttpException e = null;
        List<HttpHost> hosts = hostsFor(stid, hid, logger);
        for (int i = 0; i < hosts.size(); ++i) {
            try (CloseableHttpResponse response =
                storageClient.sendStorageRequest(
                    hosts.get(i),
                    query,
                    logger,
                    headers))
            {
                long start = System.currentTimeMillis();
                DataExtractor dataExtractor;
                if (hid == null) {
                    dataExtractor = this.dataExtractor;
                } else {
                    dataExtractor = DataType.C2C.dataExtractor();
                }
                StorageData storageData =
                    dataExtractor.extractData(response, extractorConfig);
                if (storageData == null) {
                    throw new UnsupportedMediaTypeException(
                        "No data extracted");
                }
                StringBuilder sb = new StringBuilder();
                storageData.toStringBuilder(sb);
                sb.append(" extracted in ");
                sb.append(System.currentTimeMillis() - start);
                sb.append(" ms");
                logger.info(new String(sb));
                return storageData;
            } catch (HttpException | IOException ex) {
                if (e == null) {
                    e = HttpExceptionConverter.toHttpException(ex);
                } else {
                    e.addSuppressed(ex);
                }
            }
        }
        if (e == null) {
            e = new NotFoundException("No hosts resolved for " + stid);
        } else if (tvm2RenewalTask != null) {
            Throwable cause = e.getCause();
            // unwrap possibly wrapped 403 Forbidden exception
            if (cause instanceof BadResponseException) {
                e = (HttpException) cause;
            }
        }
        throw e;
    }

    @Override
    public void rotateLogs(final HttpContext context) throws HttpException {
        super.rotateLogs(context);
        jniWrapper.logrotate();
    }

    @Override
    public void start() throws IOException {
        if (mulcagateClient != null) {
            reactor.start();
            mulcagateClient.start();
        }
        if (tvm2RenewalTask != null) {
            tvm2RenewalTask.start();
        }
        super.start();
    }

    @Override
    @SuppressWarnings("try")
    public void close() throws IOException {
        try (GenericAutoCloseableChain<IOException> chain = this.chain) {
            if (tvm2RenewalTask != null) {
                tvm2RenewalTask.cancel();
            }
            super.close();
        }
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        StorageData storageData;
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            try {
                storageData = new StorageData(
                    CharsetUtils.contentType(entity),
                    CharsetUtils.toDecodable(entity));
            } catch (IOException e) {
                throw HttpExceptionConverter.toHttpException(e);
            }
        } else {
            storageData = sendStorageRequest(request, context);
        }
        String result;
        try {
            processingLock.lock();
            try {
                result = storageData.processWith(
                    request.getRequestLine().getUri(),
                    jniWrapper);
            } finally {
                processingLock.unlock();
            }
        } catch (JniWrapperUnprocessableInputException e) {
            throw new UnsupportedMediaTypeException(e);
        } catch (JniWrapperException e) {
            throw new ServiceUnavailableException(e);
        }
        response.setEntity(
            new StringEntity(
                result,
                cokemulatorConfig.contentType().withCharset(
                    CharsetUtils.acceptedCharset(request))));
    }

    public ImmutableCokemulatorConfig cokemulatorConfig() {
        return cokemulatorConfig;
    }

    public ImmutableDataExtractorConfig extractorConfig() {
        return extractorConfig;
    }

    public DataExtractor dataExtractor() {
        return dataExtractor;
    }

    public Tvm2TicketRenewalTask tvm2RenewalTask() {
        return tvm2RenewalTask;
    }

    public StorageClient tikaiteClient() {
        return tikaiteClient;
    }

    public MulcagateClient mulcagateClient() {
        return mulcagateClient;
    }
}

