package ru.yandex.cokemulator;

import org.apache.http.HttpHost;
import org.apache.http.entity.ContentType;

import ru.yandex.client.tvm2.ImmutableTvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfig;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.server.ImmutableHttpProxyConfig;
import ru.yandex.jniwrapper.ImmutableJniWrapperConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.util.storage.ImmutableDataExtractorConfig;

public class ImmutableCokemulatorConfig
    extends ImmutableHttpProxyConfig
    implements CokemulatorConfig
{
    private final ImmutableHttpHostConfig storageConfig;
    private final HttpHost avaHost;
    private final int concurrency;
    private final ImmutableJniWrapperConfig jniWrapperConfig;
    private final DataType dataType;
    private final ImmutableDataExtractorConfig dataExtractorConfig;
    private final ContentType contentType;
    private final String uriSuffix;
    private final ImmutableHttpHostConfig mulcagateConfig;
    private final int cokemulatorPort;
    private final ImmutableHttpHostConfig tikaiteConfig;
    private final ImmutableTvm2ClientConfig tvm2ClientConfig;
    private final String apeClientId;
    private final String unistorageClientId;
    private final String uriPrefix;

    public ImmutableCokemulatorConfig(final CokemulatorConfig config)
        throws ConfigException
    {
        super(config);
        storageConfig = STORAGE_CONFIG.validate(config.storageConfig());
        concurrency = CONCURRENCY.validate(config.concurrency());
        jniWrapperConfig =
            JNIWRAPPER_CONFIG.validate(config.jniWrapperConfig());
        dataType = config.dataType();
        dataExtractorConfig =
            new ImmutableDataExtractorConfig(config.dataExtractorConfig());
        contentType = config.contentType();
        uriSuffix = config.uriSuffix();
        uriPrefix = config.uriPrefix();
        avaHost = config.avaHost();

        if (contentType == null) {
            throw new ConfigException("Content type is not set");
        }

        switch (dataType) {
            case C2C:
                HttpHostConfig mulcagateConfig = config.mulcagateConfig();
                if (mulcagateConfig == null) {
                    throw new ConfigException("Mulcagate not configured");
                }
                this.mulcagateConfig =
                    new ImmutableHttpHostConfig(mulcagateConfig);
                this.cokemulatorPort =
                    COKEMULATOR_PORT.validate(config.cokemulatorPort());
                this.tikaiteConfig = null;
                this.tvm2ClientConfig = null;
                apeClientId = null;
                unistorageClientId = null;
                break;
            case C2SRW:
                this.mulcagateConfig = null;
                this.cokemulatorPort = 0;
                this.tikaiteConfig = null;
                Tvm2ClientConfig tvm2ClientConfig = config.tvm2ClientConfig();
                if (tvm2ClientConfig == null) {
                    throw new ConfigException("TVM2 not configured");
                }
                this.tvm2ClientConfig =
                    new ImmutableTvm2ClientConfig(tvm2ClientConfig);
                apeClientId = APE_CLIENT_ID.validate(config.apeClientId());
                unistorageClientId =
                    UNISTORAGE_CLIENT_ID.validate(config.unistorageClientId());
                break;
            default:
                this.mulcagateConfig = null;
                this.cokemulatorPort = 0;
                HttpHostConfig tikaiteConfig = config.tikaiteConfig();
                if (tikaiteConfig == null) {
                    this.tikaiteConfig = null;
                } else {
                    this.tikaiteConfig =
                        new ImmutableHttpHostConfig(tikaiteConfig);
                }
                this.tvm2ClientConfig = null;
                apeClientId = null;
                unistorageClientId = null;
                break;
        }
    }

    @Override
    public ImmutableHttpHostConfig storageConfig() {
        return storageConfig;
    }

    @Override
    public int concurrency() {
        return concurrency;
    }

    @Override
    public ImmutableJniWrapperConfig jniWrapperConfig() {
        return jniWrapperConfig;
    }

    @Override
    public DataType dataType() {
        return dataType;
    }

    @Override
    public ImmutableDataExtractorConfig dataExtractorConfig() {
        return dataExtractorConfig;
    }

    @Override
    public ContentType contentType() {
        return contentType;
    }

    @Override
    public String uriSuffix() {
        return uriSuffix;
    }

    @Override
    public ImmutableHttpHostConfig mulcagateConfig() {
        return mulcagateConfig;
    }

    @Override
    public int cokemulatorPort() {
        return cokemulatorPort;
    }

    @Override
    public ImmutableHttpHostConfig tikaiteConfig() {
        return tikaiteConfig;
    }

    @Override
    public ImmutableTvm2ClientConfig tvm2ClientConfig() {
        return tvm2ClientConfig;
    }

    @Override
    public String apeClientId() {
        return apeClientId;
    }

    @Override
    public String unistorageClientId() {
        return unistorageClientId;
    }

    @Override
    public String uriPrefix() {
        return uriPrefix;
    }

    @Override
    public HttpHost avaHost() {
        return avaHost;
    }
}

