package ru.yandex.executor.concurrent;

import java.util.ArrayList;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.concurrent.FutureBase;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.util.timesource.TimeSource;

public class ConcurrentExecutor<T>
    extends Thread
    implements GenericAutoCloseable<RuntimeException>
{
    private final Queue<RetryTask<T>> retryTasks = new PriorityQueue<>();
    private final TasksProvider<T> tasksProvider;
    private final TasksExecutor<T> tasksExecutor;
    private final int concurrencyLevel;
    private final int maxRetryTasksCount;
    private final Lock lock;
    private final Condition cond;
    private volatile boolean closed = false;
    private int onAir = 0;

    public ConcurrentExecutor(
        final ThreadGroup threadGroup,
        final String threadName,
        final TasksProvider<T> tasksProvider,
        final TasksExecutor<T> tasksExecutor,
        final int concurrencyLevel,
        final int maxRetryTasksCount)
    {
        super(threadGroup, threadName);
        this.tasksProvider = tasksProvider;
        this.tasksExecutor = tasksExecutor;
        this.concurrencyLevel = concurrencyLevel;
        this.maxRetryTasksCount = maxRetryTasksCount;
        lock = tasksProvider.lock();
        cond = lock.newCondition();
        setDaemon(true);
    }

    private void taskCompleted() {
        lock.lock();
        try {
            --onAir;
            cond.signal();
        } finally {
            lock.unlock();
        }
    }

    private void taskFailed(final T task, final Exception e) {
        long retryInterval = tasksExecutor.retryIntervalFor(e);
        if (retryInterval >= 0L) {
            // Since we're rescheduling task, we need to interrupt getTask()
            long retryAt =
                TimeSource.INSTANCE.currentTimeMillis() + retryInterval;
            lock.lock();
            try {
                retryTasks.add(new RetryTask<>(task, retryAt));
                --onAir;
                interrupt();
            } finally {
                lock.unlock();
            }
        } else {
            // No new task in retryTasks
            // Just signal in case anybody waiting for available onAir task
            lock.lock();
            try {
                --onAir;
                cond.signal();
            } finally {
                lock.unlock();
            }
        }
    }

    public ConcurrentExecutorStats stats() {
        int onAir;
        int retryTasks;
        lock.lock();
        try {
            onAir = this.onAir;
            retryTasks = this.retryTasks.size();
        } finally {
            lock.unlock();
        }
        return new ConcurrentExecutorStats(onAir, retryTasks);
    }

    @Override
    public void close() {
        lock.lock();
        try {
            closed = true;
            interrupt();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void run() {
        List<T> tasks = new ArrayList<>(concurrencyLevel);
        while (!closed) {
            tasks.clear();
            long now = TimeSource.INSTANCE.currentTimeMillis();
            int retryTasksSize;
            int size = 0;
            // First take out retry tasks if any
            lock.lock();
            try {
                retryTasksSize = retryTasks.size();
                if (retryTasksSize > 0) {
                    while (size + onAir < concurrencyLevel) {
                        RetryTask<T> retryTask = retryTasks.peek();
                        if (retryTask == null || retryTask.retryAt > now) {
                            break;
                        } else {
                            tasks.add(retryTasks.poll().task);
                            ++size;
                        }
                    }
                }
                // If there is enough space for additional tasks, fetch them
                int maxNewTasks = Math.min(
                    maxRetryTasksCount - retryTasksSize,
                    concurrencyLevel - onAir - size);
                for (int i = 0; i < maxNewTasks; ++i) {
                    T task = tasksProvider.tryGetTask();
                    if (task == null) {
                        // No tasks avaiable
                        break;
                    } else {
                        tasks.add(task);
                        ++size;
                    }
                }
                if (size == 0) {
                    if (onAir >= concurrencyLevel) {
                        // No space for new tasks, wait for
                        // interrupt or signal
                        cond.await(100L, TimeUnit.MILLISECONDS);
                    } else {
                        RetryTask<T> retryTask = retryTasks.peek();
                        if (retryTask == null) {
                            // no retry tasks available
                            // wait for any task available or interrupt
                            T task = tasksProvider.getTask();
                            if (task == null) {
                                // TasksProvider terminated
                                return;
                            } else {
                                retryTasks.add(new RetryTask<>(task, 0L));
                            }
                        } else {
                            // There is some space for additional tasks
                            // Wait for nearest retry task or signal
                            long sleepInterval = retryTask.retryAt - now;
                            if (sleepInterval > 0L) {
                                cond.await(
                                    sleepInterval,
                                    TimeUnit.MILLISECONDS);
                            }
                        }
                    }
                } else {
                    onAir += size;
                }
            } catch (InterruptedException e) {
                // This is fine.
            } finally {
                lock.unlock();
            }
            for (int i = 0; i < size; ++i) {
                T task = tasks.get(i);
                tasksExecutor.execute(task, new TaskContext(task));
            }
        }
    }

    private static class RetryTask<T> implements Comparable<RetryTask<T>> {
        private final T task;
        private final long retryAt;

        RetryTask(final T task, final long retryAt) {
            this.task = task;
            this.retryAt = retryAt;
        }

        @Override
        public int compareTo(final RetryTask<T> task) {
            return Long.compare(retryAt, task.retryAt);
        }
    }

    private class TaskContext
        extends FutureBase<Void>
        implements FutureCallback<Void>
    {
        private final T task;

        public TaskContext(final T task) {
            this.task = task;
        }

        @Override
        protected boolean onCancel(final boolean mayInterruptIfRunning) {
            taskCompleted();
            return super.onCancel(mayInterruptIfRunning);
        }

        @Override
        protected void onComplete(final Void result) {
            taskCompleted();
        }

        @Override
        protected void onFailure(final Exception e) {
            taskFailed(task, e);
        }

        @Override
        public void completed(final Void result) {
            completedInternal(result);
        }
    }
}

