package ru.yandex.executor.concurrent;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BinaryOperator;

import ru.yandex.concurrent.ParallelHashMap;
import ru.yandex.function.GenericAutoCloseable;

// TasksProvider which holds counters associated with some keys, provides tasks
// when requested and merge counters if counter with the same key wasn't
// removed yet
public class CountersTasksProvider<K, V>
    implements GenericAutoCloseable<RuntimeException>, TasksProvider<V>
{
    private final Lock lock = new ReentrantLock();
    private final Condition cond = lock.newCondition();
    private final BinaryOperator<V> countersMerger;
    private final ParallelHashMap<K, V> tasks;
    private volatile boolean closed = false;

    public CountersTasksProvider(
        final BinaryOperator<V> countersMerger,
        final int concurrency)
    {
        this.countersMerger = countersMerger;
        tasks = new ParallelHashMap<>(concurrency);
    }

    // Add new counters or update existing
    // Returns true if new task was added, false on existing counters merge
    public boolean addCounters(final K key, final V counters) {
        V oldCounters = tasks.putIfAbsent(key, counters, countersMerger);
        if (oldCounters == null) {
            // new task inserted successfully, notify getTask()
            lock.lock();
            try {
                cond.signal();
            } finally {
                lock.unlock();
            }
            return true;
        } else {
            return false;
        }
    }

    public int size() {
        return tasks.size();
    }

    @Override
    public void close() {
        closed = true;
        lock.lock();
        try {
            cond.signal();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public Lock lock() {
        return lock;
    }

    // Lock already acquired
    @Override
    public V getTask() throws InterruptedException {
        while (!closed) {
            V value = tasks.removeAnyValue();
            if (value != null) {
                return value;
            }
            cond.await(100L, TimeUnit.MILLISECONDS);
        }
        return null;
    }

    @Override
    public V tryGetTask() {
        return tasks.removeAnyValue();
    }
}

