#include "composer.hpp"
#include "options.hpp"

#include <butil/datetime/date_utils.h>

#include <string>
#include <algorithm>
#include <boost/format.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/asio/ip/host_name.hpp>
#include "exp_tokens.hpp"

namespace dsn {

composer::composer(const Options& options) :
    options_(options),
    my_domain_(boost::asio::ip::host_name())
{}

void composer::my_domain(const std::string& domain)
{
    my_domain_ = domain;
}

namespace {

class domain_matcher
{
public:
    explicit domain_matcher(const std::string &address)
    {
        std::string::size_type pos = address.find('@');
        if (pos != std::string::npos)
            domain_ = boost::trim_copy(address.substr(pos + 1));
        else
            domain_ = boost::trim_copy(address);
    }

    bool operator ()(const Template &t) const
    {
        return match(t.domains.begin(), t.domains.end());
    }

private:
    template <class Iterator>
    bool match(Iterator first, Iterator last) const
    {
        for (std::string::size_type p = domain_.find('.');
                p != std::string::npos; p = domain_.find('.', p + 1))
        {
            for (Iterator it = first; it != last; ++it)
                if (domain_.compare(p + 1, domain_.length() - p - 1, *it) == 0)
                    return true;
        }
        return false;
    }

    std::string domain_;
};

template <typename Range>
typename Range::const_iterator find_template(
    const Range &templates, const std::string &domain)
{
    typename Range::const_iterator it = std::find_if(
        templates.begin(), templates.end(), domain_matcher(domain));
    if (it == templates.end())
        it = std::find_if(templates.begin(), templates.end(),
            domain_matcher(".com"));
    return it;
}

std::string get_status_code(const std::string &str)
{
    std::string::size_type pos1 = str.find(' ');
    if (pos1 == std::string::npos)
        return "";
    std::string::size_type pos2 = str.find(' ', pos1 + 1);
    if (pos2 == std::string::npos)
        return "";
    return str.substr(pos1 + 1, pos2 - pos1 - 1);
}

// Decode "xtext" string as specified in section 5 of the RFC 1891.
std::string decode_xtext(const std::string &s)
{
    std::string result;
    for (std::string::size_type i = 0; i < s.size(); i++)
    {
        if (s[i] == '+')
        {
            if (i + 2 < s.size()
                    && std::isxdigit(s[i + 1]) && std::isxdigit(s[i + 2]))
            {
                result.push_back(
                    std::strtol(s.substr(i + 1, 2).c_str(), NULL, 16));
                i += 2;
            }
            else
                return result; // Malformed xtext.
        }
        else if (s[i] == '=')
            return result; // Malformed xtext.
        else
            result.push_back(s[i]);
    }
    return result;
}

// A bunch of parameters for make_dsn() routine.
struct param_t
{
    std::string host_;
    std::string sender_;
    std::vector<Template>::const_iterator template_;
    int m_mode;
    std::string my_domain_;
};

std::pair<bool, std::string> make_dsn(
    const std::string& id,
    const std::string& orig_id,
    const std::string& sender,
    const std::vector<rcpt>& rcpts,
    param_t &params)
{
    std::string delivery_errors;
    std::ostringstream report;

    // Add per-message fields.
    report << "Reporting-MTA: dns; " << params.host_ << "\r\n";
    if (!orig_id.empty())
        report << "Original-Envelope-Id: "
            << decode_xtext(orig_id) << "\r\n";
    report << "\r\n";

    bool have_rcpts = false;
    for (const auto& r : rcpts)
    {
        if ((params.m_mode == Options::FAILURE
                && ((r.notify_mode_ & Options::FAILURE) || (r.notify_mode_ == Options::NONE))
                && r.status_ == rcpt::failure)
            || (params.m_mode == Options::SUCCESS
                && (r.notify_mode_ & Options::SUCCESS)
                && r.status_ == rcpt::success))
        {
            const std::string remote_answer = boost::trim_right_copy(
                r.remote_answer_);

            // Add human-readable error description.
            delivery_errors += str(boost::format(
                "<%1%> host %2% said: %3% (in reply to RCPT command)\r\n")
                % r.name_ % params.host_ % remote_answer);

            // Add per-recipient fields.
            report
                << "Final-Recipient: rfc822; " << r.name_ << "\r\n"
                << "Original-Recipient: rfc822; " << r.name_ << "\r\n"
                << "Action: " << (r.status_ == rcpt::success
                    ? "delivered" : "failed") << "\r\n"
                << "Status: " << get_status_code(r.remote_answer_) << "\r\n"
                << "Diagnostic-Code: smtp; " << remote_answer << "\r\n"
                << "\r\n";

            have_rcpts = true;
        }
    }

    if (!have_rcpts)
        return {false, {}};

    exp_tokens::token_list_t tokens;
    tokens["errors"] = delivery_errors;
    tokens["myhostname"] = params.host_;
    tokens["mydomain"] = params.my_domain_;

    const std::string boundary = id + "/" + boost::asio::ip::host_name();

    // Add MIME headers.
    std::ostringstream headers;
    headers
        << "Date: " << DateUtils::rfc2822time(std::time(NULL), {}) << "\r\n"
        << "From: " << params.sender_ << "\r\n"
        << "To: " << sender << "\r\n"
        << "Auto-Submitted: auto-replied\r\n"
        << "MIME-Version: 1.0\r\n"
        << "Content-Type: multipart/report; report-type=delivery-status;\r\n"
        << "\t boundary=\"" << boundary << "\"\r\n"
        << "Message-Id: <" << id << "@" << boost::asio::ip::host_name() << ">\r\n"
        << "Content-Transfer-Encoding: 8bit\r\n"
        << "Subject: " << params.template_->subject << "\r\n"
        << "\r\n";
    std::string message = headers.str();

    // Add human-readable explanation part.
    message += "--" + boundary + "\r\n";
    message += "Content-Type: text/plain";
    if (!params.template_->charset.empty())
        message += "; charset=" + params.template_->charset;
    message += "\r\n\r\n";
    message += exp_tokens::expand_tokens(params.template_->body, tokens);
    message += "\r\n";

    // Add report part.
    message += "--" + boundary + "\r\n";
    message += "Content-Type: message/delivery-status\r\n\r\n";
    message += report.str();

    // Add closing boundary.
    message += "--" + boundary + "--\r\n";

    return {true, message};
}

} // namespace

std::pair<bool, std::string> composer::compose(
    type_t t,
    const std::string& id,
    const std::string& orig_id,
    const std::string& sender,
    const std::vector<rcpt>& rcpts) const
{
    const std::string& host = boost::asio::ip::host_name();

    // Pick a DSN template for specific notification type and sender domain.
    typedef std::vector<Template> template_container;
    typedef template_container::const_iterator template_iterator;
    const template_container& templates = (t == success)
        ? options_.successTemplates : options_.failureTemplates;
    template_iterator it = find_template(templates, sender);
    if (it == templates.end())
        throw std::runtime_error("Failed to pick a DSN template");

    // Compose an envelope.
    param_t param;
    param.host_ = host;
    param.sender_ = options_.origin;
    param.template_ = it;
    param.m_mode = (t == success) ? Options::SUCCESS : Options::FAILURE;
    param.my_domain_ = my_domain_;
    return make_dsn(id, orig_id, sender, rcpts, param);
}

} // namespace dsn
