#include "conf_parser.hpp"
#include "options.hpp"

#include <boost/asio.hpp>
#include <boost/program_options.hpp>
#include <boost/tokenizer.hpp>

namespace dsn {

static void split_template(const std::string &rawTemplate, Template &tmplt) {
    // parse domains list
    std::string::size_type lf_pos = rawTemplate.find('\n');
    if (lf_pos != std::string::npos) {
        std::string domains(rawTemplate, 0, lf_pos - 1);

        boost::char_separator<char> separator(" ,");
        boost::tokenizer<boost::char_separator<char> > tokens(domains, separator);
        std::copy(tokens.begin(), tokens.end(), std::back_inserter(tmplt.domains));
    }

    // parse headers
    auto begin = rawTemplate.begin(), end = rawTemplate.end(), pos = begin;
    while (pos != end) {
        char c = *pos;
        if (c == '\n') {
            ++pos;
            break;
        } else if (c == '\r' && pos + 1 != end && *(pos + 1) == '\n') {
            pos += 2;
            break;
        }

        if (!isspace(c)) {
            std::string::const_iterator name_begin = pos;
            while (pos != end && (*pos != ':' && !isspace(*pos)))
                ++pos;
            std::string::const_iterator name_end = pos;
            if (pos == end)
                break;

            if (*pos != ':') {
                pos = name_begin;

                while (pos != end && *pos++ != '\n');
            } else {
                ++pos; // skip ':' character

                while (pos != end && (*pos == ' ' || *pos == '\t'))
                    ++pos;

                std::string::const_iterator value_begin = pos, value_end = pos;
                while (pos != end) {
                    value_end = pos;

                    while (pos != end) {
                        c = *pos;

                        // Check for end of line
                        if (c == '\r' && pos + 1 != end && *(pos + 1) == '\n') {
                            value_end = pos;
                            pos += 2;
                            break;
                        } else if (c == '\n') {
                            value_end = pos;
                            ++pos;
                            break;
                        }

                        ++pos;
                    }

                    if (pos == end)
                        break;
                    c = *pos;

                    // handle the case of folded lines
                    if (!(c == ' ' || c == '\t'))
                        break;

                    // Check for end of contents
                    if (c == '\r' && pos + 1 != end && *(pos + 1) == '\n') {
                        pos += 2;
                        break;
                    } else if (c == '\n') {
                        ++pos;
                        break;
                    }
                }

                std::string header_name(name_begin, name_end);
                if (strcasecmp(header_name.c_str(), "subject") == 0)
                    tmplt.subject.assign(value_begin, value_end);
                if (strcasecmp(header_name.c_str(), "charset") == 0)
                    tmplt.charset.assign(value_begin, value_end);
            }
        }
        else
            while (pos != end && *pos++ != '\n'); // skip error and advance to the next line
    }

    if (pos != end)
        tmplt.body.assign(pos, end);
}

template <class InputIterator, class OutputIterator>
static void split_template(InputIterator first, InputIterator last, OutputIterator result)
{
    while (first != last)
    {
        Template t;
        split_template(*first++, t);
        *result++ = t;
    }
}

int parse_mode(const std::string &mode)
{
    int result = Options::NEVER;
    boost::tokenizer<> tokens(mode);
    for (boost::tokenizer<>::const_iterator i = tokens.begin(); i != tokens.end(); ++i)
    {
        if (strcasecmp(i->c_str(), "none") == 0)
            return Options::NEVER;
        if (strcasecmp(i->c_str(), "success") == 0)
            result |= Options::SUCCESS;
        else if (strcasecmp(i->c_str(), "failure") == 0)
            result |= Options::FAILURE;
    }
    return result;
}

void Options::init(const yplatform::ptree& pt) {
    origin = pt.get("origin", std::string("mailer_daemon@") + boost::asio::ip::host_name());

    if (!pt.count("config_file"))
        throw std::invalid_argument("required DSN options not specified");

    auto configPath = pt.get<std::string>("config_file");
    if (!configPath.empty()) {
        typedef std::vector<std::string> RawTemplateList;

        boost::program_options::options_description configOptions("DSN");
        boost::program_options::variables_map configVariables;

        configOptions.add_options()
            ("dsn_fault_template", boost::program_options::value<RawTemplateList>(),
                "template for DSN failure message")
            ("dsn_success_template", boost::program_options::value<RawTemplateList>(),
                "template for DSN success message");

        store(parse_dsn_config_file(configPath.c_str(), configOptions, true), configVariables);
        notify(configVariables);

        if (configVariables.count("dsn_fault_template")) {
            const auto & fails = configVariables["dsn_fault_template"].as<RawTemplateList>();
            split_template(fails.begin(), fails.end(), std::back_inserter(failureTemplates));
        }
        if (configVariables.count("dsn_success_template")) {
            const auto & success = configVariables["dsn_success_template"].as<RawTemplateList>();
            split_template(success.begin(), success.end(), std::back_inserter(successTemplates));
        }
    }
}

}   // namespace dsn
