package ru.yandex.parser.email;

import java.util.ArrayList;
import java.util.List;

import org.apache.james.mime4j.dom.address.DomainList;
import org.apache.james.mime4j.dom.address.Mailbox;

import ru.yandex.function.CharArrayProcessor;
import ru.yandex.parser.rfc2047.Rfc2047DecodersProvider;
import ru.yandex.parser.rfc2047.Rfc2047Parser;
import ru.yandex.util.match.tables.MatchTables;

public class EmailParser
    implements CharArrayProcessor<List<Mailbox>, RuntimeException>
{
    private static final char[] EMPTY_CBUF = new char[0];
    private static final boolean[] ADDRESS_LIST_DELIMITERS =
        MatchTables.createMatchTable(",:@<");
    private static final boolean[] ROUTE_DELIMITERS =
        MatchTables.createMatchTable(">,:");
    private static final boolean[] MAILBOX_ADDRESS_DELIMITERS =
        MatchTables.createMatchTable("@>");
    private static final boolean[] MAILBOX_LIST_DELIMITERS =
        MatchTables.createMatchTable(",;@<");
    private static final boolean[] MAILBOX_LIST_DOMAIN_DELIMITERS =
        MatchTables.createMatchTable(",;");
    private static final boolean[] RFC2047_CHARSET_CHARS =
        MatchTables.createMatchTable(
            "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz"
            + "1234567890+.:_-");

    private final StringBuilder sb = new StringBuilder();
    private final List<String> domainList = new ArrayList<>();
    private final List<Mailbox> addresses = new ArrayList<>();
    private char[] cbuf = EMPTY_CBUF;
    private Rfc2047DecodersProvider decoders = null;

    private void ensureCBufCapacity(final int len) {
        if (cbuf.length < len) {
            cbuf = new char[Math.max(len, cbuf.length << 1)];
        }
    }

    // Parse rfc2047 encoded addresses, like
    // "=?UTF-8?B?0Jwu0JLQuNC00LXQvg==?=" <mvideo@sender.mvideo.ru>
    public List<Mailbox> parse(
        final String value,
        final Rfc2047DecodersProvider decoders)
    {
        this.decoders = decoders;
        try {
            return doParse(value);
        } finally {
            this.decoders = null;
        }
    }

    // Parse decoded addresses, like ones provided by fastsrv:
    // "Дмитрий Потапов" <analizer@yandex.ru>
    public List<Mailbox> parseDecoded(final String value) {
        return doParse(value);
    }

    private List<Mailbox> doParse(final String value) {
        int len = value.length();
        ensureCBufCapacity(len);
        value.getChars(0, len, cbuf, 0);
        return doProcess(cbuf, 0, len);
    }

    @Override
    public List<Mailbox> process(final char[] cbuf) {
        return process(cbuf, 0, cbuf.length);
    }

    @Override
    public List<Mailbox> process(
        final char[] buf,
        final int off,
        final int len)
    {
        return doProcess(buf, off, len);
    }

    private List<Mailbox> doProcess(
        final char[] buf,
        final int off,
        final int len)
    {
        addresses.clear();
        int end = off + len;
        for (int i = off; i < end;) {
            char c = buf[i];
            if (c == ',') {
                ++i;
            } else {
                i = parseAddress(buf, i, end);
            }
        }
        return addresses;
    }

    private String sbToString() {
        if (sb.length() > 0) {
            return sb.toString();
        } else {
            return "";
        }
    }

    private int parseAddress(
        final char[] buf,
        final int start,
        final int end)
    {
        int pos = parseValue(ADDRESS_LIST_DELIMITERS, buf, start, end);
        if (pos == end) {
            if (sb.length() > 0) {
                createMailbox(sb.toString());
            }
            return end;
        }
        char c = buf[pos];
        switch (c) {
            case '<':
                pos = parseMailboxAddress(sbToString(), buf, pos + 1, end);
                break;
            case '@':
                String opening = sbToString();
                pos = parseDomain(',', buf, pos + 1, end);
                addresses.add(new Mailbox(null, null, opening, sbToString()));
                break;
            case ':':
                pos = parseMailboxes(buf, pos + 1, end);
                if (pos < end && buf[pos] == ';') {
                    ++pos;
                }
                break;
            default:
                if (sb.length() > 0) {
                    createMailbox(sb.toString());
                }
                break;
        }
        return pos;
    }

    private int parseValue(
        final boolean[] delimiters,
        final char[] buf,
        final int start,
        final int end)
    {
        sb.setLength(0);
        boolean whitespace = false;

        for (int i = start; i < end;) {
            char c = buf[i];
            switch (c) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                    i = skipWhitespace(buf, i + 1, end);
                    whitespace = true;
                    break;
                case '(':
                    i = skipComment(buf, i + 1, end);
                    break;
                case '"':
                    if (whitespace && sb.length() > 0) {
                        sb.append(' ');
                    }
                    whitespace = false;
                    i = copyQuoted(buf, i + 1, end);
                    break;
                default:
                    if (MatchTables.match(delimiters, c)) {
                        return i;
                    } else {
                        if (whitespace && sb.length() > 0) {
                            sb.append(' ');
                        }
                        whitespace = false;
                        sb.append(c);
                        if (c == '=') {
                            // XXX: Deviation from mime4j, parse rfc2047
                            // encoded value, to be consistent with Thunderbird
                            // and GMail
                            int pos = tryCopyRfc2047(buf, i + 1, end);
                            if (pos != -1) {
                                i = pos;
                                break;
                            }
                        }
                        i = copyUnquoted(delimiters, buf, i + 1, end);
                        break;
                    }
            }
        }
        return end;
    }

    private static int skipWhitespace(
        final char[] buf,
        final int start,
        final int end)
    {
        for (int i = start; i < end; ++i) {
            switch (buf[i]) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                    break;
                default:
                    return i;
            }
        }
        return end;
    }

    private static int skipComment(
        final char[] buf,
        final int start,
        final int end)
    {
        int level = 1;
        boolean escaped = false;
        for (int i = start; i < end; ++i) {
            if (escaped) {
                escaped = false;
            } else {
                switch (buf[i]) {
                    case '\\':
                        escaped = true;
                        break;
                    case '(':
                        ++level;
                        break;
                    case ')':
                        if (--level == 0) {
                            return i + 1;
                        }
                        break;
                    default:
                        break;
                }
            }
        }
        return end;
    }

    private int copy(
        final char delimiter,
        final char[] buf,
        final int start,
        final int end)
    {
        for (int i = start; i < end; ++i) {
            char c = buf[i];
            switch (c) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                case '(':
                    sb.append(buf, start, i - start);
                    return i;
                default:
                    if (c == delimiter) {
                        sb.append(buf, start, i - start);
                        return i;
                    }
                    break;
            }
        }
        sb.append(buf, start, end - start);
        return end;
    }

    private int copy(
        final boolean[] delimiters,
        final char[] buf,
        final int start,
        final int end)
    {
        for (int i = start; i < end; ++i) {
            char c = buf[i];
            switch (c) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                case '(':
                    sb.append(buf, start, i - start);
                    return i;
                default:
                    if (MatchTables.match(delimiters, c)) {
                        sb.append(buf, start, i - start);
                        return i;
                    }
                    break;
            }
        }
        sb.append(buf, start, end - start);
        return end;
    }

    private int copyQuoted(
        final char[] buf,
        final int start,
        final int end)
    {
        boolean escaped = false;
        for (int i = start; i < end; ++i) {
            char c = buf[i];
            if (escaped) {
                escaped = false;
                if (c != '"' && c != '\\') {
                    sb.append('\\');
                }
                sb.append(c);
            } else {
                switch (c) {
                    case '"':
                        return i + 1;
                    case '\\':
                        escaped = true;
                        break;
                    case '\r':
                    case '\n':
                        break;
                    default:
                        sb.append(c);
                        break;
                }
            }
        }
        return end;
    }

    private int copyUnquoted(
        final boolean[] delimiters,
        final char[] buf,
        final int start,
        final int end)
    {
        for (int i = start; i < end; ++i) {
            char c = buf[i];
            switch (c) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                case '(':
                case '"':
                case '=':
                    sb.append(buf, start, i - start);
                    return i;
                default:
                    if (MatchTables.match(delimiters, c)) {
                        sb.append(buf, start, i - start);
                        return i;
                    }
                    break;
            }
        }
        sb.append(buf, start, end - start);
        return end;
    }

    private int tryCopyRfc2047(
        final char[] buf,
        final int start,
        final int end)
    {
        if (start < end && buf[start] == '?') {
            int pos = start + 1;
            while (pos < end) {
                char c = buf[pos];
                if (MatchTables.match(RFC2047_CHARSET_CHARS, c)) {
                    ++pos;
                } else {
                    break;
                }
            }
            if (pos < end && buf[pos++] == '?' && pos < end) {
                switch (buf[pos++]) {
                    case 'B':
                    case 'b':
                    case 'Q':
                    case 'q':
                        if (pos < end && buf[pos++] == '?') {
                            while (pos < end) {
                                char c = buf[pos];
                                if (c == '?'
                                    && pos + 1 < end
                                    && buf[pos + 1] == '=')
                                {
                                    pos += 2;
                                    sb.append(buf, start, pos - start);
                                    return pos;
                                } else {
                                    ++pos;
                                }
                            }
                        }
                        break;
                    default:
                        break;
                }
            }
        }
        return -1;
    }

    private Mailbox createMailbox(
        final String opening,
        final String localPart,
        final String domain)
    {
        DomainList domainList;
        if (this.domainList.isEmpty()) {
            domainList = null;
        } else {
            domainList = new DomainList(this.domainList);
        }
        String name;
        if (opening == null) {
            name = null;
        } else if (decoders == null) {
            name = opening;
        } else {
            name = Rfc2047Parser.decode(opening, decoders);
        }
        return new Mailbox(name, domainList, localPart, domain);
    }

    private void createMailbox(final String localPart) {
        if (decoders == null) {
            addresses.add(new Mailbox(null, null, localPart, null));
        } else {
            // This is direct violation of rfc2047 §5(3), but GMail does this
            String decoded = Rfc2047Parser.decode(localPart, decoders);
            int idx = decoded.indexOf('@');
            if (idx != -1) {
                addresses.addAll(new EmailParser().parseDecoded(decoded));
            } else {
                addresses.add(new Mailbox(null, null, decoded, null));
            }
        }
    }

    private int parseMailboxAddress(
        final String opening,
        final char[] buf,
        final int start,
        final int end)
    {
        int pos = parseRoute(buf, start, end);
        pos = parseValue(MAILBOX_ADDRESS_DELIMITERS, buf, pos, end);
        String localPart = sbToString();
        if (pos == end) {
            addresses.add(createMailbox(opening, localPart, null));
            return pos;
        }
        if (buf[pos] != '@') { // i.e. buf[pos] = '>'
            addresses.add(createMailbox(opening, localPart, null));
            // XXX: Deviation from mime4j, increment pos to swallow '>'
            return pos + 1;
        }
        pos = parseDomain('>', buf, pos + 1, end);
        String domain = sbToString();
        if (pos == end) {
            addresses.add(createMailbox(opening, localPart, domain));
            return pos;
        }
        // pos != end ⇒ buf[pos] == '>'
        ++pos;
        while (pos < end) {
            switch (buf[pos]) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                    pos = skipWhitespace(buf, pos + 1, end);
                    break;
                case '(':
                    pos = skipComment(buf, pos + 1, end);
                    break;
                default:
                    addresses.add(createMailbox(opening, localPart, domain));
                    return pos;
            }
        }
        addresses.add(createMailbox(opening, localPart, domain));
        return end;
    }

    private int parseRoute(final char[] buf, final int start, final int end) {
        domainList.clear();
        int i = start;
        while (true) {
            i = skipAllWhitespace(buf, i, end);
            if (i == end) {
                return i;
            }
            if (buf[i] == '@') {
                ++i;
            } else {
                break;
            }
            i = parseDomain(ROUTE_DELIMITERS, buf, i, end);
            if (sb.length() > 0) {
                domainList.add(sb.toString());
            }
            if (i == end) {
                break;
            }
            char c = buf[i];
            if (c == ',') {
                ++i;
            } else {
                if (c == ':') {
                    ++i;
                }
                break;
            }
        }
        return i;
    }

    private int parseDomain(
        final char delimiter,
        final char[] buf,
        final int start,
        final int end)
    {
        sb.setLength(0);
        for (int i = start; i < end;) {
            char c = buf[i];
            switch (c) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                    i = skipWhitespace(buf, i + 1, end);
                    break;
                case '(':
                    i = skipComment(buf, i + 1, end);
                    break;
                default:
                    if (c == delimiter) {
                        return i;
                    } else {
                        i = copy(delimiter, buf, i, end);
                    }
                    break;
            }
        }
        return end;
    }

    private int parseDomain(
        final boolean[] delimiters,
        final char[] buf,
        final int start,
        final int end)
    {
        sb.setLength(0);
        for (int i = start; i < end;) {
            char c = buf[i];
            switch (c) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                    i = skipWhitespace(buf, i + 1, end);
                    break;
                case '(':
                    i = skipComment(buf, i + 1, end);
                    break;
                default:
                    if (MatchTables.match(delimiters, c)) {
                        return i;
                    } else {
                        i = copy(delimiters, buf, i, end);
                    }
                    break;
            }
        }
        return end;
    }

    private static int skipAllWhitespace(
        final char[] buf,
        final int start,
        final int end)
    {
        for (int i = start; i < end;) {
            switch (buf[i]) {
                case ' ':
                case '\t':
                case '\r':
                case '\n':
                    i = skipWhitespace(buf, i + 1, end);
                    break;
                case '(':
                    i = skipComment(buf, i + 1, end);
                    break;
                default:
                    return i;
            }
        }
        return end;
    }

    private int parseMailboxes(
        final char[] buf,
        final int start,
        final int end)
    {
        for (int i = start; i < end;) {
            char c = buf[i];
            if (c == ';') {
                return i;
            }
            if (c == ',') {
                ++i;
            } else {
                i = parseMailbox(buf, i, end);
            }
        }
        return end;
    }

    private int parseMailbox(
        final char[] buf,
        final int start,
        final int end)
    {
        int pos = parseValue(MAILBOX_LIST_DELIMITERS, buf, start, end);
        if (pos == end) {
            if (sb.length() > 0) {
                createMailbox(sb.toString());
            }
            return end;
        }
        char c = buf[pos];
        if (c == '<') {
            pos = parseMailboxAddress(sbToString(), buf, pos + 1, end);
        } else if (c == '@') {
            String localPart = sbToString();
            pos =
                parseDomain(MAILBOX_LIST_DOMAIN_DELIMITERS, buf, pos + 1, end);
            addresses.add(new Mailbox(null, null, localPart, sbToString()));
        } else {
            if (sb.length() > 0) {
                createMailbox(sb.toString());
            }
        }
        return pos;
    }
}

