package ru.yandex.parser.email;

import java.util.AbstractMap;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.detect.locale.LocaleDetector;
import ru.yandex.util.string.StringUtils;

public class MailAliases {
    public static final MailAliases INSTANCE = new MailAliases();

    private static final String DASH = "-";

    private final Map<String, MailService> domainToService = new HashMap<>();

    protected MailAliases() {
        for (MailService service: new MailService[]{
            new YandexMailService(
                "yandex.ru",
                "narod.ru",
                "ya.ru",
                "yandex.by",
                "yandex.com",
                "yandex.com.tr",
                "yandex.kz",
                "yandex.az",
                "yandex.com.am",
                "yandex.com.ge",
                "yandex.co.il",
                "yandex.kg",
                "yandex.lt",
                "yandex.lv",
                "yandex.md",
                "yandex.tj",
                "yandex.tm",
                "yandex.uz",
                "yandex.fr",
                "yandex.ee",
                "xn--d1acpjx3f.xn--p1ai"), // яндекс.рф
            new YandexMailService(
                "yandex-team.ru",
                "mail.yandex-team.ru",
                "yandex-team.com",
                "mail.yandex-team.com",
                "ld.yandex.ru"),
            new MailMailService(),
            new GmailMailService()})
        {
            for (String alias: service.aliases()) {
                domainToService.put(alias, service);
            }
        }
    }

    // looks for '@' and checks if it is not a first or last character and
    // there is a period after '@' and it is not a last character etc.
    public static int emailSeparatorPos(final String email) {
        int atPos = email.lastIndexOf('@');
        if (atPos > 0) {
            int periodPos = email.indexOf('.', atPos + 1);
            if (periodPos > atPos + 1 && periodPos + 1 < email.length()) {
                return atPos;
            }
        }
        return -1;
    }

    public boolean containsDomain(final String email) {
        int sep = emailSeparatorPos(email);
        if (sep == -1) {
            return domainToService.containsKey(email);
        }

        String domain = domain(email, sep);
        return domainToService.containsKey(domain);
    }

    public static String domain(final String email, final int sep) {
        return domain(email.substring(sep + 1));
    }

    public static String domain(final String domain) {
        return LocaleDetector.INSTANCE.toLowerCase(domain);
    }

    public String normalizeEmail(final String email) {
        int sep = emailSeparatorPos(email);
        if (sep == -1) {
            return email;
        } else {
            return normalizeEmail(email, sep);
        }
    }

    public String normalizeEmail(final String email, final int sep) {
        Map.Entry<String, String> parsed = parseAndNormalize(email, sep);
        return StringUtils.concat(parsed.getKey(), '@', parsed.getValue());
    }

    public Map.Entry<String, String> parseAndNormalize(final String email) {
        int sep = emailSeparatorPos(email);
        if (sep == -1) {
            return new AbstractMap.SimpleImmutableEntry<>(email, null);
        } else {
            return parseAndNormalize(email, sep);
        }
    }

    public Map.Entry<String, String> parseAndNormalize(
        final String email,
        final int sep)
    {
        return parseAndNormalize(email.substring(0, sep), domain(email, sep));
    }

    public Map.Entry<String, String> parseAndNormalize(
        String user,
        String domain)
    {
        MailService service =
            domainToService.getOrDefault(domain, DefaultMailService.INSTANCE);
        domain = service.normalizeDomain(domain);
        if (!service.caseSensitiveUserName()) {
            user = LocaleDetector.INSTANCE.toLowerCase(user);
        }
        if (service.supportPlusTags()) {
            int sep = user.indexOf('+', 1);
            if (sep != -1) {
                user = user.substring(0, sep);
            }
        }

        user = service.normalizeUserName(user);

        return new AbstractMap.SimpleImmutableEntry<>(user, domain);
    }

    public String extractAndNormalizeDomain(final String email) {
        int sep = emailSeparatorPos(email);
        if (sep == -1) {
            return email;
        }
        return normalizeDomain(domain(email, sep));
    }

    public String normalizeDomain(final String domain) {
        return domainToService
            .getOrDefault(domain, DefaultMailService.INSTANCE)
            .normalizeDomain(domain);
    }

    public void addEquivalentEmails(
        final Set<String> emails,
        final String email)
    {
        int sep = emailSeparatorPos(email);
        if (sep == -1) {
            emails.add(email);
        } else {
            addEquivalentEmails(emails, email, sep);
        }
    }

    // it is expected that emailSeparatorPos > 0, i.e. passed email looks valid
    public void addEquivalentEmails(
        final Set<String> emails,
        final String email,
        final int emailSeparatorPos)
    {
        String user = email.substring(0, emailSeparatorPos);
        String domain = domain(email, emailSeparatorPos);
        emails.add(user + '@' + domain);
        MailService service =
            domainToService.getOrDefault(domain, DefaultMailService.INSTANCE);
        Set<String> userNames = new HashSet<>();
        userNames.add(user);
        userNames.add(service.normalizeUserName(user));
        if (!service.caseSensitiveUserName()) {
            for (String name: new HashSet<>(userNames)) {
                userNames.add(LocaleDetector.INSTANCE.toLowerCase(name));
            }
        }
        if (service.supportPlusTags()) {
            for (String name: new HashSet<>(userNames)) {
                int sep = name.indexOf('+', 1);
                if (sep != -1) {
                    userNames.add(name.substring(0, sep));
                }
            }
        }
        for (String alias: service.aliases()) {
            for (String name: userNames) {
                emails.add(name + '@' + alias);
            }
        }
    }

    public Set<String> equivalentEmails(final String email) {
        int emailSeparatorPos = emailSeparatorPos(email);
        if (emailSeparatorPos == -1) {
            return Collections.singleton(email);
        } else {
            return equivalentEmails(email, emailSeparatorPos);
        }
    }

    // it is expected that emailSeparatorPos > 0, i.e. passed email looks valid
    public Set<String> equivalentEmails(
        final String email,
        final int emailSeparatorPos)
    {
        Set<String> emails = new HashSet<>();
        addEquivalentEmails(emails, email, emailSeparatorPos);
        return emails;
    }

    private interface MailService {
        List<String> aliases();

        String normalizeDomain(String domain);

        default boolean caseSensitiveUserName() {
            return false;
        }

        default boolean supportPlusTags() {
            return false;
        }

        default String normalizeUserName(final String user) {
            return user;
        }
    }

    private enum DefaultMailService implements MailService {
        INSTANCE;

        @Override
        public List<String> aliases() {
            return Collections.emptyList();
        }

        @Override
        public String normalizeDomain(final String domain) {
            return domain;
        }
    }

    private static class BasicMailService implements MailService {
        private final List<String> aliases;
        private final String domain;

        BasicMailService(final String domain) {
            this.domain = domain;
            aliases = Collections.singletonList(domain);
        }

        BasicMailService(final String... aliases) {
            this.aliases = Arrays.asList(aliases);
            domain = aliases[0];
        }

        @Override
        public List<String> aliases() {
            return aliases;
        }

        @Override
        public String normalizeDomain(final String domain) {
            return this.domain;
        }
    }

    private static class YandexMailService extends BasicMailService {
        YandexMailService(final String... domains) {
            super(domains);
        }

        @Override
        public boolean supportPlusTags() {
            return true;
        }

        private static boolean allDigitsOrDashes(final String str) {
            for (int i = 1; i < str.length(); ++i) {
                char c = str.charAt(i);
                if (c != '-' && !Character.isDigit(c)) {
                    return false;
                }
            }
            return true;
        }

        @Override
        public String normalizeUserName(String user) {
            user = user.replace('.', '-');
            char c = user.charAt(0);
            if (c == '+' && user.length() > 1) {
                if (allDigitsOrDashes(user)) {
                    user = user.substring(1).replace(DASH, "");
                }
            } else if (Character.isDigit(c)) {
                if (allDigitsOrDashes(user)) {
                    user = user.replace(DASH, "");
                }
            }
            return user;
        }
    }

    private static class MailMailService extends BasicMailService {
        MailMailService() {
            super("mail.ru");
        }

        @Override
        public boolean supportPlusTags() {
            return true;
        }
    }

    private static class GmailMailService extends BasicMailService {
        GmailMailService() {
            super("gmail.com", "googlemail.com");
        }

        @Override
        public boolean supportPlusTags() {
            return true;
        }

        @Override
        public String normalizeUserName(final String user) {
            return user.replace(".", "");
        }
    }
}

