package ru.yandex.geocoder;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.google.protobuf.ExtensionRegistry;
import com.google.protobuf.InvalidProtocolBufferException;
import org.apache.http.HttpException;
import yandex.maps.proto.common2.geo_object.GeoObjectOuterClass.GeoObject;
import yandex.maps.proto.common2.geometry.GeometryOuterClass;
import yandex.maps.proto.common2.metadata.MetadataOuterClass.Metadata;
import yandex.maps.proto.common2.response.ResponseOuterClass.Response;
import yandex.maps.proto.search.address.AddressOuterClass.Address;
import yandex.maps.proto.search.address.AddressOuterClass.Component;
import yandex.maps.proto.search.business.Business;
import yandex.maps.proto.search.business_internal.BusinessInternal;
import yandex.maps.proto.search.business_internal.BusinessInternal.CompanyInfo;
import yandex.maps.proto.search.geocoder.Geocoder;
import yandex.maps.proto.search.geocoder_internal.GeocoderInternal;
import yandex.maps.proto.search.geocoder_internal.GeocoderInternal.ToponymInfo;
import yandex.maps.proto.search.kind.KindOuterClass.Kind;

import ru.yandex.http.util.BadRequestException;

public class GeocoderResultBuilder implements GeocoderResult {
    private final ExtensionRegistry registry = ExtensionRegistry.newInstance();

    private int size = 0;
    private int geoid = -1;

    private String countryName = "";

    private List<String> cities = new ArrayList<>();
    private List<GeometryOuterClass.BoundingBox> boundedBy = new ArrayList<>();

    public GeocoderResultBuilder() {
        registry.add(Geocoder.gEOOBJECTMETADATA);
        registry.add(GeocoderInternal.tOPONYMINFO);
        registry.add(Business.gEOOBJECTMETADATA);
        registry.add(BusinessInternal.cOMPANYINFO);
    }

    public GeocoderResult build() {
        return new ImmutableGeocoderResult(this);
    }

    public void parseGeocoderFromProto(final byte[] buf) throws HttpException {
        try {
            Response response = Response.parseFrom(buf, registry);
            List<GeoObject> geoObjects = parseGeoObjects(response);

            if (geoObjects == null || geoObjects.isEmpty()) {
                return;
            }

            GeoObject firstGeoObject = geoObjects.get(0);
            Address address = parseAddress(firstGeoObject);
            countryName = parseCountry(address);
            geoid = parseGeoid(firstGeoObject);

            for (GeoObject geoObject: geoObjects) {
                if (geoObject != null && geoObject.hasBoundedBy()) {
                    boundedBy.add(geoObject.getBoundedBy());
                }
                address = parseAddress(geoObject);
                String city = parseCity(address);
                if (!city.isEmpty()) {
                    cities.add(city);
                }
            }
            size = boundedBy.size();
        } catch (InvalidProtocolBufferException e) {
            throw new BadRequestException(
                "Can't parse protobuf message: " + e.getMessage(),
                e);
        }
    }

    private List<GeoObject> parseGeoObjects(final Response response) {
        if (response != null && response.hasReply()) {
            return response.getReply().getGeoObjectList();
        } else {
            return Collections.emptyList();
        }
    }

    private Geocoder.GeoObjectMetadata parseGeocoderGeoObjectMetadata(
        final GeoObject geoObject)
    {
        if (geoObject != null) {
            for (Metadata metadata: geoObject.getMetadataList()) {
                if (metadata.hasExtension(Geocoder.gEOOBJECTMETADATA)) {
                    return metadata.getExtension(Geocoder.gEOOBJECTMETADATA);
                }
            }
        }
        return null;
    }

    private Business.GeoObjectMetadata parseBusinessGeoObjectMetadata(
        final GeoObject geoObject)
    {
        if (geoObject != null) {
            for (Metadata metadata: geoObject.getMetadataList()) {
                if (metadata.hasExtension(Business.gEOOBJECTMETADATA)) {
                    return metadata.getExtension(Business.gEOOBJECTMETADATA);
                }
            }
        }
        return null;
    }

    private ToponymInfo parseToponymInfo(final GeoObject geoObject) {
        Geocoder.GeoObjectMetadata metadata =
            parseGeocoderGeoObjectMetadata(geoObject);
        if (metadata != null
            && metadata.hasExtension(GeocoderInternal.tOPONYMINFO))
        {
            return metadata.getExtension(GeocoderInternal.tOPONYMINFO);
        }
        return null;
    }

    private CompanyInfo parseCompanyInfo(final GeoObject geoObject) {
        Business.GeoObjectMetadata metadata =
            parseBusinessGeoObjectMetadata(geoObject);
        if (metadata != null
            && metadata.hasExtension(BusinessInternal.cOMPANYINFO))
        {
            return metadata.getExtension(BusinessInternal.cOMPANYINFO);
        }
        return null;
    }

    private Address parseAddress(final GeoObject geoObject) {
        Address result = null;
        Geocoder.GeoObjectMetadata geoMetadata =
            parseGeocoderGeoObjectMetadata(geoObject);
        if (geoMetadata != null && geoMetadata.hasAddress()) {
            result = geoMetadata.getAddress();
        }
        if (result == null) {
            Business.GeoObjectMetadata bizMetadata =
                parseBusinessGeoObjectMetadata(geoObject);
            if (bizMetadata != null && bizMetadata.hasAddress()) {
                result = bizMetadata.getAddress();
            }
        }
        return result;
    }

    private String parseAddressComponent(
        final Address address,
        final Kind kind)
    {
        if (address != null) {
            for (Component c: address.getComponentList()) {
                if (c.hasName() && c.getKindList().contains(kind)) {
                    return c.getName();
                }
            }
        }
        return "";
    }

    private String parseCity(final Address address) {
        return parseAddressComponent(address, Kind.LOCALITY);
    }

    private String parseCountry(final Address address) {
        return parseAddressComponent(address, Kind.COUNTRY);
    }

    private int parseGeoid(final GeoObject geoObject) {
        int result = -1;
        ToponymInfo toponymInfo = parseToponymInfo(geoObject);
        if (toponymInfo != null && toponymInfo.hasGeoid()) {
            result = toponymInfo.getGeoid();
        }
        if (result == -1) {
            CompanyInfo companyInfo = parseCompanyInfo(geoObject);
            if (companyInfo != null && companyInfo.hasGeoid()) {
                result = companyInfo.getGeoid();
            }
        }
        return result;
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public double lowerLatitude(final int index) {
        return boundedBy.get(index).getLowerCorner().getLat();
    }

    @Override
    public double lowerLongitude(final int index) {
        return boundedBy.get(index).getLowerCorner().getLon();
    }

    @Override
    public double upperLatitude(final int index) {
        return boundedBy.get(index).getUpperCorner().getLat();
    }

    @Override
    public double upperLongitude(final int index) {
        return boundedBy.get(index).getUpperCorner().getLon();
    }

    @Override
    public int getGeoid() { return geoid; }

    @Override
    public String getCountryName() { return countryName; }

    @Override
    public List<String> getCities() { return cities; }
}

