package ru.yandex.geocoder;

import java.util.ArrayList;
import java.util.List;

public class ImmutableGeocoderResult implements GeocoderResult {
    private static final int LONGITUDE = 0;
    private static final int LATITUDE = 1;

    private static final double MAX_DIAGONAL = 100.0; // km
    private static final double RADIOUS = 6371.0; // km

    private final int size;
    private final double[][][] boundedBy;
    private final int geoid;
    private final String countryName;
    private final List<String> cities;

    public ImmutableGeocoderResult(final GeocoderResult result) {
        List<double[][]> boundedBy = new ArrayList<>(result.size());
        for (int i = 0; i < result.size(); ++i) {
            if (i != 0 && diagonal(result, i) > MAX_DIAGONAL) {
                continue;
            }
            double[] latitudes =
                Coordinates.Latitude.split(
                    result.lowerLatitude(i),
                    result.upperLatitude(i));
            double[] longitudes =
                Coordinates.Longitude.split(
                    result.lowerLongitude(i),
                    result.upperLongitude(i));
            for (int iLat = 0; iLat + 1 < latitudes.length; iLat += 2) {
                for (int iLong = 0; iLong + 1 < longitudes.length; iLong += 2) {
                    double[][] current =
                        new double[POINTS_NUMBER][COORDINATES_NUMBER];
                    current[0][LATITUDE] = latitudes[iLat];
                    current[1][LATITUDE] = latitudes[iLat + 1];
                    current[0][LONGITUDE] = longitudes[iLong];
                    current[1][LONGITUDE] = longitudes[iLong + 1];
                    boundedBy.add(current);
                }
            }
        }
        this.size = boundedBy.size();
        this.boundedBy = boundedBy.toArray(new double[this.size][][]);
        this.geoid = result.getGeoid();
        this.countryName = result.getCountryName();
        this.cities = result.getCities();
    }

    private double diagonal(final GeocoderResult result, final int index) {
        double lat1 = Math.toRadians(result.lowerLatitude(index));
        double lat2 = Math.toRadians(result.upperLatitude(index));
        double lon1 = Math.toRadians(result.lowerLongitude(index));
        double lon2 = Math.toRadians(result.upperLongitude(index));
        return RADIOUS * Math.acos(
            Math.sin(lat1) * Math.sin(lat2)
            + Math.cos(lat1) * Math.cos(lat2)
                * Math.cos(lon1 - lon2));
    }

    @Override
    public int size() {
        return size;
    }

    @Override
    public double lowerLatitude(final int index) {
        return boundedBy[index][0][LATITUDE];
    }

    @Override
    public double upperLatitude(final int index) {
        return boundedBy[index][1][LATITUDE];
    }

    @Override
    public double lowerLongitude(final int index) {
        return boundedBy[index][0][LONGITUDE];
    }

    @Override
    public double upperLongitude(final int index) {
        return boundedBy[index][1][LONGITUDE];
    }

    @Override
    public int getGeoid() {
        return geoid;
    }

    @Override
    public String getCountryName() {
        return countryName;
    }

    @Override
    public List<String> getCities() {
        return cities;
    }
}
